import { configureStore } from '@reduxjs/toolkit';
import thunk from 'redux-thunk';
import { middleware as localStorageMiddleware } from 'modules/localStorage';
import createReducer from 'reducers';
import promiseMiddleware from 'middlewares/promises';
import fetchMiddleware from 'middlewares/fetch';
import apiMiddleware from 'middlewares/api';
import logger from 'middlewares/logger';
import websocket from 'middlewares/websocket';
import { AppStore } from './types';
import { withAsyncReducer } from './withAsyncReducer';
import { withLocalStorageReducer } from './withLocalStorageReducer';
import { defaultStoreData } from './defaultStoreDataForUnitTest';

declare global {
  interface Window {
    reduxStore: AppStore;
  }
}

const preloadedState =
  process.env.STORYBOOK_ENV === 'test' || process.env.NODE_ENV === 'test'
    ? defaultStoreData
    : undefined;

export function createStore(): AppStore {
  const store = configureStore({
    preloadedState,
    reducer: createReducer(),
    middleware: [
      logger,
      localStorageMiddleware,
      apiMiddleware,
      fetchMiddleware,
      promiseMiddleware,
      thunk,
      websocket,
    ],
    devTools: true,
  });

  return withLocalStorageReducer(withAsyncReducer(store, { createReducer }));
}

window.reduxStore = createStore();

export default window.reduxStore;
