import { Subject } from 'rxjs';
import { debounceTime } from 'rxjs/operators';
import { EnhancedStore, Reducer } from '@reduxjs/toolkit';
import { WithAsyncReducers } from './types';

export type AsyncReducers = { [key: string]: Reducer };

export const REMOVE_REDUCER_CLEAR_STATE = 'REMOVE_REDUCER_CLEAR_STATE';

export const withClearStateReducer = (reducer) => (state, action) => {
  if (action.type === REMOVE_REDUCER_CLEAR_STATE) {
    const newState = { ...state };
    action.payload.forEach((name) => {
      delete newState[name];
    });

    return newState;
  }

  return reducer(state, action);
};

export const withAsyncReducer = (
  reduxStore: EnhancedStore,
  {
    createReducer,
  }: {
    createReducer: (asyncReducers?: AsyncReducers) => Reducer;
  },
) => {
  const store = Object.create(reduxStore) as EnhancedStore & WithAsyncReducers;

  const namesForCleanup = new Set<string>();

  const changeEvent$ = new Subject<void>();

  let asyncReducers: { [key: string]: Reducer } = {};

  const clearState = () => {
    store.dispatch({
      type: REMOVE_REDUCER_CLEAR_STATE,
      payload: Array.from(namesForCleanup),
    });
  };

  const updateReducer = () => {
    store.replaceReducer(createReducer(asyncReducers));
  };

  const clearReducer = () => {
    const reducers = { ...asyncReducers };
    namesForCleanup.forEach((name) => {
      delete reducers[name];
    });

    asyncReducers = reducers;

    updateReducer();
  };

  const clearStore = () => {
    if (!namesForCleanup.size) {
      return;
    }

    clearReducer();
    clearState();
    namesForCleanup.clear();
  };

  changeEvent$.pipe(debounceTime(1000)).subscribe(clearStore);

  store.injectReducer = function(name, reducer, forceUpdate = false): void {
    namesForCleanup.delete(name);
    changeEvent$.next();

    if (!asyncReducers[name] || forceUpdate) {
      asyncReducers[name] = reducer;
      updateReducer();
    }
  };

  store.replaceReducer = function(reducer) {
    reduxStore.replaceReducer(withClearStateReducer(reducer));
  };

  store.removeReducer = function(name: string) {
    namesForCleanup.add(name);
    changeEvent$.next();
  };

  return store;
};
