import React from 'react';
import PropTypes from 'prop-types';
import { Field, FieldArray } from 'redux-form';
import FieldDatePicker from 'requestForms/components/FieldDatePicker';
import RenderDatePicker from 'requestForms/components/RenderDatePicker';
import FieldTreeSelect from 'requestForms/components/FieldTreeSelect';
import { TagsField } from 'modules/tags/components/TagsField';
import cx from 'classnames';
import legoReduxFormComponents from 'lego/redux-form';
import FieldArrayComponent from './FieldArray';
import ClientSelect from './ClientSelect';
import QueueSelect from './QueueSelect';
import CategorySelect from './CategorySelect';
import Subject from './Subject';
import FieldWrap from './FieldWrap';
import css from './styles.modules.scss';

const FIELD_ARRAY = {
  fieldArray: FieldArrayComponent,
  Tags: TagsField,
};

const COMPONENTS = {
  datePicker: RenderDatePicker,
  client: ClientSelect,
  queue: QueueSelect,
  category: CategorySelect,
  treeSelect: FieldTreeSelect,
  subject: Subject,
  ...legoReduxFormComponents,
  ...FIELD_ARRAY,
};

const MODAL_CONTROL = {
  client: true,
  queue: true,
  category: true,
};

const DEFAULTS = {
  textarea: { maxLength: 1000 },
  select: { mode: 'radio-check' },
};

const getComponent = (component, isString) => {
  if (!isString) {
    return component;
  }

  return COMPONENTS[component] || component;
};

const getComponentProps = (component, isString) => {
  if (!isString) {
    return {};
  }

  if (legoReduxFormComponents[component]) {
    const props = { size: 'm' };

    if (component === 'CheckBox') {
      props.lineHeightSize = 'm';
    }

    return props;
  }

  return DEFAULTS[component] || {};
};

const BaseRow = (props) => {
  const {
    className,
    fieldClassName,
    view,
    name,
    component,
    placeholder,
    label,
    hint,
    required,
    wrapProps,
    content,
    children,
    fieldWidth,
    props: fieldProps,
    ...rest
  } = props;
  const isString = typeof component === 'string';
  let RenderField;

  if (component === 'datePicker') {
    RenderField = FieldDatePicker;
  } else if (FIELD_ARRAY[component]) {
    RenderField = FieldArray;
  } else {
    RenderField = Field;
  }

  return (
    <FieldWrap
      className={className}
      view={view}
      name={name}
      label={label}
      hint={hint}
      required={required}
      wrapProps={wrapProps}
      content={content}
      fieldWidth={fieldWidth}
    >
      <RenderField
        className={cx(fieldClassName, css.control, {
          [css.control_modal]: MODAL_CONTROL[component],
        })}
        id={name}
        name={name}
        component={getComponent(component, isString)}
        classNameInput={css.input}
        placeholder={placeholder || label}
        label={label}
        {...getComponentProps(component, isString)}
        {...rest}
        {...fieldProps}
      >
        {children}
      </RenderField>
    </FieldWrap>
  );
};

BaseRow.propTypes = {
  className: PropTypes.string,
  fieldClassName: PropTypes.string,
  children: PropTypes.node,
  content: PropTypes.node,
  view: PropTypes.oneOf(['row']),
  name: PropTypes.string,
  component: PropTypes.oneOfType([PropTypes.string, PropTypes.func, PropTypes.element]),
  placeholder: PropTypes.string,
  label: PropTypes.string,
  hint: PropTypes.string,
  required: PropTypes.bool,
  wrapProps: PropTypes.instanceOf(Object),
  props: PropTypes.instanceOf(Object),
  fieldWidth: PropTypes.string,
};

BaseRow.defaultProps = {
  className: undefined,
  fieldClassName: undefined,
  children: undefined,
  content: undefined,
  view: 'row',
  name: undefined,
  component: 'input',
  placeholder: undefined,
  label: undefined,
  hint: undefined,
  required: false,
  wrapProps: {},
  props: undefined,
  fieldWidth: 'max',
};

export default BaseRow;
