// TODO: https://st.yandex-team.ru/CRM-10289
/* eslint-disable no-restricted-globals */

import React, { RefObject, MutableRefObject, ReactNode } from 'react';
import Prompt from 'components/Prompt';
import { FormApi, FieldValidator } from 'final-form';
import { Field, FormRenderProps } from 'react-final-form';
import { AsyncFormContext } from 'lego/hoc/asyncField';
import Button from '@crm/components/dist/lego2/Button';
import ButtonToggle from 'lego/components/ButtonToggle';
import cx from 'classnames';
import SuggestStaff from 'lego/final-form/SuggestStaff';
import AttachFiles from 'lego/final-form/AttachFiles';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import { EditInfoPanel } from 'components/EditInfoPanel';
import { Toolbar2 } from 'components/Toolbar2';
import {
  AttachFilesViewerField,
  AttachFilesButton,
  AttachFilesDND,
  AttachFilesViewerInputRef,
} from 'components/AttachFiles';
import { config } from 'services/Config';
import { AttachFilesDNDContainer } from 'components/AttachFiles';
import { MergeRef } from 'utils/mergeRef';
import { RollerWithPopup } from 'components/Roller';
import { FormWithLoading, composeValidators } from 'components/FinalForm/withLoadingMutator';
import { ToolTipWrapper } from '@crm/components/dist/lego2/Tooltip';
import lang from 'constants/lang';
import Icon from 'lego/components/Icon';
import * as keyset from './Comment.i18n';
import css from './styles.modules.scss';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nSend = commonI18n('send');
const i18nCancel = commonI18n('cancel');
const i18nComment = i18n('comment');
const i18nActionConfirm = i18n('actionConfirm');
const i18nInviteToComments = i18n('inviteToComments');
const i18nEmptyFormError = i18n('emptyFormError');

const formValidate = (values) => {
  const error: { [key: string]: string } = {};

  if (!values.text && !(values.files && values.files.files && values.files.files.length)) {
    error._error = i18nEmptyFormError; // eslint-disable-line no-underscore-dangle
  }

  return error;
};

const filterValues = ({ files, ...other }) => ({
  ...other,
});

const inviteeIconProvider = (iconCls) => (
  <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="person_add" />
);

export type ActionsPosition = 'left' | 'right';

export interface RenderProps {
  placeholder: string;
  autoFocus: boolean;
  files: React.ReactNode | null;
  actions: React.ReactNode;
  inviteesField?: React.ReactNode;
  inviteeToggle?: React.ReactNode;
  toolbar: React.ReactNode;
  refTextComponent?: React.Ref<unknown>;
  addonToolbarBefore?: React.ReactNode;
  addonToolbarAfter?: React.ReactNode;
  addonToolbarRightAfter?: React.ReactNode;
  view?: 'clear';
  actionsPosition?: ActionsPosition;
  controlRef?: React.Ref<unknown>;
  maxLength?: number;
}

export type Render = (props: RenderProps) => React.ReactNode;

export interface Props {
  className?: string;
  objId?: number;
  containerId?: number;
  okButtonLabel: string;
  sendAndOpenButton: boolean;
  cancelButtonLabel: string;
  cancelButton: boolean;
  onCancelClick: (event: unknown) => {};
  editInfoText?: string;
  onClearEditClick?: () => void;
  resetOnCancel: boolean;
  filesObjectName: string;
  placeholder: string;
  focusOnMount: boolean;
  files: boolean;
  dispatch: () => {};
  onSubmit: (values: unknown) => {};
  form: string;
  isConfirmCancel?: boolean;
  isConfirmRouting?: boolean;
  isInvitees?: boolean;
  size?: LegoSizeProp;
  onSuccess?: () => void;
  refTextComponent?: React.Ref<unknown>;
  render: Render;
  initialValues?: FormData;
  addonBefore?: React.ReactNode;
  addonAfter?: React.ReactNode;
  addonToolbarBefore?: React.ReactNode;
  addonToolbarAfter?: React.ReactNode;
  addonToolbarRightAfter?: React.ReactNode;
  view?: 'clear';
  actionsPosition?: ActionsPosition;
  formApiRef?: RefObject<unknown>;
  fileInputRef: RefObject<unknown>;
  controlRef?: React.Ref<unknown>;
  maxLength?: number;
  validate?: FieldValidator<unknown>;
  forceUseFeatureIssueTimelineV2?: boolean;
}

interface State {
  isInviteesField: boolean;
}

interface FormData {
  invitees?: unknown[];
}

class CommentBase extends React.Component<Props, State> {
  public static defaultProps = {
    className: undefined,
    objId: undefined,
    containerId: undefined,
    okButtonLabel: i18nSend,
    sendAndOpenButton: false,
    cancelButtonLabel: i18nCancel,
    editInfoText: undefined,
    onCancelClick: undefined,
    onClearEditClick: undefined,
    cancelButton: false,
    resetOnCancel: false,
    filesObjectName: 'container',
    placeholder: i18nComment,
    focusOnMount: false,
    isConfirmCancel: true,
    /** @deprecated use isPreventUnload from withPreventUnload */
    isConfirmRouting: false,
    isInvitees: false,
    files: true,
    size: 's',
    onSuccess: undefined,
    render: undefined,
    actionsPosition: 'left',
    forceUseFeatureIssueTimelineV2: false,
  };

  private buttonNode: HTMLButtonElement | null = null;

  private form: FormApi<FormData>;

  private finalFormSubmit: FormRenderProps['handleSubmit'];

  private issueWillBeMovedToOpen: boolean;

  private fileInputRef = new MergeRef<AttachFilesViewerInputRef>();
  private formRef = React.createRef<HTMLFormElement>();

  public constructor(props) {
    super(props);

    const { initialValues } = this.props;
    this.state = {
      isInviteesField: Boolean(
        initialValues && Array.isArray(initialValues.invitees) && initialValues.invitees.length,
      ),
    };
  }

  private handleToggleInvitees = () => {
    const { isInviteesField } = this.state;

    this.setState({ isInviteesField: !isInviteesField });
  };

  private doSubmit = async (event?) => {
    await this.finalFormSubmit(event);
    this.handleSendSuccess();
  };

  private handleSend = (values) => {
    const promise = this.props.onSubmit(
      filterValues({
        ...values,
        moveToOpen: this.issueWillBeMovedToOpen,
      }),
    );

    this.issueWillBeMovedToOpen = false;
    return promise;
  };

  private handleSendSuccess = () => {
    if (this.props.onSuccess) {
      this.props.onSuccess();
    }
    this.form.reset();
  };

  private handleCancelClick = (e) => {
    if (this.props.onCancelClick) {
      this.props.onCancelClick(e);
    }

    if (this.props.resetOnCancel) {
      this.form.reset();
    }
  };

  private handleClearEditClick = () => {
    if (this.props.onClearEditClick) {
      this.props.onClearEditClick();
    }
  };

  private handleSendAndOpenClick = () => {
    this.issueWillBeMovedToOpen = true;
    this.doSubmit();
  };

  private handleKeyDown = (e) => {
    switch (e.key) {
      case 'Enter':
        if (e.ctrlKey || e.metaKey) {
          e.preventDefault();
          if (this.buttonNode) {
            /*
             *  похоже баг в redux-form:
             *  когда форма отправляется по ctrl + enter поле комментария в фокусе
             *  redux-form некорректно обрабатывает эту ситуацию: сначала срабатывает reset формы,
             *  потом onBlur поля комментария с текущим содержимым, из-за чего текст визуально не очищается
             *
             *  фикс такой: при отправке уводим фокус на кнопку отправить
             * */
            this.buttonNode.focus();
            this.buttonNode.dispatchEvent(new MouseEvent('click'));
          }
        }
        break;
      case 'Escape': {
        e.preventDefault();
        const { isConfirmCancel, onCancelClick, resetOnCancel } = this.props;
        const { pristine } = this.form.getState();
        if (
          (onCancelClick || resetOnCancel) &&
          (!isConfirmCancel ||
            pristine ||
            (!pristine && isConfirmCancel && confirm(i18nActionConfirm)))
        ) {
          this.handleCancelClick(e);
        }
        break;
      }
      default:
    }
  };

  private getButtonNode = (node: HTMLButtonElement) => {
    this.buttonNode = node;
  };

  private getActions = (): React.ReactNode => {
    const { okButtonLabel, sendAndOpenButton, cancelButtonLabel, cancelButton, size } = this.props;

    const { pristine, submitting, valid, errors } = this.form.getState();

    const actions = (
      <React.Fragment>
        <AsyncFormContext.Consumer>
          {(async) => (
            <>
              {sendAndOpenButton && (
                <ToolTipWrapper
                  containerClassName={css.toolbar__FooterTooltipContainer}
                  state="alert"
                  content={errors?._error}
                  direction="top-end"
                  openDelay={500}
                >
                  <Button
                    size={size}
                    disabled={async || pristine || submitting || !valid}
                    onClick={this.handleSendAndOpenClick}
                    view="action"
                    className={css.toolbar__FooterTooltipItem}
                  >
                    Отправить и открыть
                  </Button>
                </ToolTipWrapper>
              )}
              <ToolTipWrapper
                containerClassName={css.toolbar__FooterTooltipContainer}
                state="alert"
                content={errors?._error}
                direction="top-end"
                openDelay={500}
              >
                <Button
                  size={size}
                  disabled={async || pristine || submitting || !valid}
                  type="submit"
                  ref={this.getButtonNode}
                  view={sendAndOpenButton ? 'default' : 'action'}
                  className={css.toolbar__FooterTooltipItem}
                >
                  {okButtonLabel}
                </Button>
              </ToolTipWrapper>
            </>
          )}
        </AsyncFormContext.Consumer>
        {cancelButton && (
          <Button size={size} view="pseudo" onClick={this.handleCancelClick}>
            {cancelButtonLabel}
          </Button>
        )}
      </React.Fragment>
    );

    return actions;
  };

  public render() {
    const {
      className,
      filesObjectName,
      placeholder,
      files,
      objId,
      containerId,
      isConfirmRouting,
      isInvitees,
      focusOnMount,
      render,
      refTextComponent,
      addonBefore,
      addonAfter,
      formApiRef,
      initialValues,
      editInfoText,
      addonToolbarBefore,
      addonToolbarAfter,
      view,
      addonToolbarRightAfter,
      actionsPosition,
      controlRef,
      fileInputRef,
      maxLength,
      validate,
      forceUseFeatureIssueTimelineV2,
    } = this.props;

    const { isInviteesField } = this.state;

    this.fileInputRef.refs = [fileInputRef as RefObject<AttachFilesViewerInputRef>];

    const issueTimelineV2 = forceUseFeatureIssueTimelineV2 || config.value.features.issueTimelineV2;

    let inviteesField: React.ReactNode = null;
    if (isInvitees && isInviteesField) {
      inviteesField = (
        <Field name="invitees" component={SuggestStaff} placeholder={i18nInviteToComments} />
      );
    }

    let inviteeToggle: React.ReactNode = null;
    if (isInvitees) {
      inviteeToggle = (
        <ButtonToggle
          view="pseudo"
          open={isInviteesField}
          onClick={this.handleToggleInvitees}
          icon={inviteeIconProvider}
        />
      );
    }

    return (
      <FormWithLoading
        onSubmit={this.handleSend}
        validate={composeValidators(formValidate, validate)}
        subscription={{ submitting: true, pristine: true, valid: true }}
        initialValues={initialValues}
      >
        {(props) => {
          const { handleSubmit, pristine, form, errors } = props;
          this.form = form;

          if (formApiRef) {
            (formApiRef as MutableRefObject<unknown>).current = form;
          }

          this.finalFormSubmit = handleSubmit;
          const actions = this.getActions();
          const toolbar = this.renderToolbar(actions);

          return (
            <AttachFilesDNDContainer>
              <form
                ref={this.formRef}
                role="presentation"
                onSubmit={this.doSubmit}
                className={cx(css.root, className, css[`root_view_${view}`])}
                onKeyDown={this.handleKeyDown}
              >
                {addonBefore}
                <Prompt
                  when={isConfirmRouting && !pristine}
                  message={lang.CHANGE_LOCATION}
                  preventCloseBrowser
                />
                {editInfoText && (
                  <EditInfoPanel text={editInfoText} onClearClick={this.handleClearEditClick} />
                )}
                {render({
                  placeholder,
                  autoFocus: focusOnMount,
                  files: files && (
                    <Field
                      className={css.files}
                      name="files"
                      component={issueTimelineV2 ? AttachFilesViewerField : AttachFiles}
                      objId={objId}
                      containerId={containerId}
                      objectName={filesObjectName}
                      fileInputRef={this.fileInputRef}
                    />
                  ),
                  actions,
                  toolbar,
                  inviteesField,
                  inviteeToggle,
                  refTextComponent,
                  addonToolbarBefore: (
                    <>
                      {addonToolbarBefore}
                      {issueTimelineV2 && files && (
                        <AttachFilesButton
                          disabled={errors.loading}
                          fileInputRef={this.fileInputRef}
                        />
                      )}
                    </>
                  ),
                  addonToolbarAfter,
                  addonToolbarRightAfter,
                  view,
                  actionsPosition,
                  controlRef,
                  maxLength,
                })}
                {addonAfter}
                {issueTimelineV2 && files && (
                  <AttachFilesDND
                    zones={[
                      {
                        text: 'Перетащите файлы сюда, чтобы добавить вложение',
                        onDrop: (files, rejected, event) => {
                          if (event?.type === 'dropFileId')
                            this.fileInputRef?.current?.updateFiles(files as number[], []);
                          else this.fileInputRef?.current?.addFiles(files as File[]);
                        },
                        disabled: errors.loading,
                      },
                    ]}
                  />
                )}
              </form>
            </AttachFilesDNDContainer>
          );
        }}
      </FormWithLoading>
    );
  }

  private renderToolbar(actions: ReactNode) {
    return (
      <Toolbar2
        className={css.toolbar}
        left={this.renderToolbarLeft(actions)}
        right={this.renderToolbarRight(actions)}
      />
    );
  }

  private renderToolbarLeft(actions: ReactNode) {
    const {
      actionsPosition,
      addonToolbarBefore,
      addonToolbarAfter,
      files,
      forceUseFeatureIssueTimelineV2,
    } = this.props;

    const { errors } = this.form.getState();

    const issueTimelineV2 = forceUseFeatureIssueTimelineV2 || config.value.features.issueTimelineV2;

    return (
      <>
        {addonToolbarBefore}
        {issueTimelineV2 && files && (
          <AttachFilesButton disabled={errors.loading} fileInputRef={this.fileInputRef} />
        )}
        {actionsPosition === 'left' && actions}
        {addonToolbarAfter}
      </>
    );
  }

  private renderToolbarRight(actions: ReactNode) {
    const { actionsPosition, addonToolbarRightAfter } = this.props;

    return (
      <>
        <RollerWithPopup
          className={css.toolbar__FooterRoller}
          rollerClassNames={{
            containerClassName: css.toolbar__FooterRollerContainer,
          }}
          popupProps={{
            containerClassName: css.toolbar__FooterRollerPopupContainer,
            scope: 'inplace',
            boundary: this.formRef,
          }}
        >
          {actionsPosition === 'right' && actions}
        </RollerWithPopup>
        {addonToolbarRightAfter}
      </>
    );
  }
}

export default CommentBase;
