import { createElement, Component } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { formValueSelector } from 'redux-form';
import { jsonApiCall } from 'api/common';
import { push } from 'modules/notificationsUI/actions';
import get from 'lodash/get';
import mapValues from 'lodash/mapValues';
import { setData, getData } from 'utils/localStorage';
import NewTicketComponent from './NewTicketComponent';
import './Notification';

const FORM_NAME = 'newTicket';

const mapState = state => {
  const selector = formValueSelector(FORM_NAME);

  return {
    queueId: get(selector(state, 'queueId'), 'id'),
  };
};

const mapDispatch = (dispatch, props) => ({
  onSubmit: values =>
    jsonApiCall({
      url: '/actions/support/ticket/save',
      data: values,
    }),
  onSubmitSuccess: data => {
    const items = data && data.items;
    if (Array.isArray(items)) {
      const action = {
        view: 'NEW_TICKET',
        theme: 'success',
      };

      if (items.length === 1) {
        const item = items[0];
        action.text = `Создан новый тикет: ${item.number}`;
        action.extra = item;
      } else if (items.length > 1) {
        action.text = `Создано ${items.length} новых тикетов`;
      }

      dispatch(push(action));
    }

    props.toggle(false);
  },
  onCancel: () => props.toggle(false),
});

const LOCAL_STORAGE_KEY = 'newTicket';

class NewTicketController extends Component {
  constructor(props) {
    super(props);

    const storedInitialValues = getData(LOCAL_STORAGE_KEY);
    this.initialValues = { ...storedInitialValues, count: '1' };
  }

  handleSubmit = ({ subjectId, count, ...otherValues }) => {
    const storeInitialValues = { queueId: otherValues.queueId, categoryId: otherValues.categoryId };
    setData(LOCAL_STORAGE_KEY, storeInitialValues);

    const newValues = mapValues(otherValues, obj => obj.id);
    if (typeof subjectId === 'string') {
      newValues.subject = subjectId;
    } else if (typeof subjectId === 'object') {
      newValues.subject = subjectId.name;
    }
    newValues.count = count;

    return this.props.onSubmit(newValues);
  };

  render() {
    const props = {
      ...this.props,
      onSubmit: this.handleSubmit,
      initialValues: this.initialValues,
      form: FORM_NAME,
    };

    return createElement(NewTicketComponent, props);
  }
}

NewTicketController.propTypes = {
  onSubmit: PropTypes.func.isRequired,
};

export default connect(
  mapState,
  mapDispatch,
)(NewTicketController);
