import Bluebird from 'bluebird';
import React from 'react';
import { RouteComponentProps, Redirect } from 'react-router-dom';

type Url = string | null;

interface WithRedirectProps<Params> {
  redirect: (props: RouteComponentProps<Params>) => Bluebird<Url>;
}

interface WithRedirectState {
  url?: Url;
}

const withRedirect = <Params extends { [K in keyof Params]?: string } = {}>(
  config: WithRedirectProps<Params>,
) => (WrappedComponent: React.ComponentType<RouteComponentProps<Params>>) => {
  class WithRedirect extends React.Component<
    RouteComponentProps<Params> & WithRedirectProps<Params>,
    WithRedirectState
  > {
    public static defaultProps = config;

    private promise?: Bluebird<Url>;

    public state: WithRedirectState = {};

    public componentDidMount(): void {
      this.promise = this.props.redirect(this.props).then(result => {
        this.setState({ url: result });

        return result;
      });
    }

    public componentWillUnmount(): void {
      if (this.promise) {
        this.promise.cancel();
      }
    }

    public render() {
      const { redirect, ...rest } = this.props;
      const { url } = this.state;

      if (url === undefined) {
        return null;
      }

      if (url) {
        return <Redirect to={url} />;
      }

      return <WrappedComponent {...rest} />;
    }
  }

  return WithRedirect;
};

export default withRedirect;
