import * as React from 'react';
import Bluebird from 'bluebird';
import { AsyncFieldContext } from 'lego/hoc/asyncField';
import * as api from './api';
import AttachFiles from './AttachFiles';
import { BackendResponse, BackendFile } from './types';

interface State {
  isFetch: boolean;
}

interface AttachFilesProps {
  files: BackendFile[];
}

export interface Props extends AttachFilesProps {
  objectName: string;
  containerId?: number;
  objId?: number;
  onChange?: (data: BackendResponse) => void;
}

export default class AttachFilesWithApi extends React.Component<Props, State> {
  public static contextType = AsyncFieldContext;

  public context: React.ContextType<typeof AsyncFieldContext>;

  public state = { isFetch: false };

  private promise: Bluebird<{}> | null;

  public componentWillUnmount() {
    this.cancelPromise();
  }

  private cancelPromise = () => {
    if (this.promise) {
      this.promise.cancel();
    }
  };

  private setFetch = (flag) => {
    this.setState({ isFetch: flag });

    if (this.context) {
      this.context.setAsync(flag);
    }
  };

  private handleResponseFinally = () => {
    this.promise = null;
    this.setFetch(false);
  };

  private handleOnRemove = (id) => {
    this.cancelPromise();
    this.setFetch(true);

    this.promise = api
      .remove({
        fileId: id,
        objectName: this.props.objectName,
        containerId: this.props.containerId,
        objId: this.props.objId,
      })
      .then(this.handleResponseData)
      .finally(this.handleResponseFinally);

    return this.promise;
  };

  private handleOnDrop = (files: File[]) => {
    if (files.length === 0) {
      return null;
    }

    this.cancelPromise();
    this.setFetch(true);

    this.promise = api
      .add({
        files,
        objectName: this.props.objectName,
        containerId: this.props.containerId,
        objId: this.props.objId,
      })
      .then(this.handleResponseData)
      .finally(this.handleResponseFinally);

    return this.promise;
  };

  public getCurrentFiles() {
    return this.props.files || [];
  }

  public updateFiles = (fileIdsToLink: number[], fileIdsToRemove: number[]) => {
    this.cancelPromise();
    this.setFetch(true);

    this.promise = api
      .update({
        fileIdsToLink,
        fileIdsToRemove,
        objectName: this.props.objectName,
        containerId: this.props.containerId,
        objId: this.props.objId,
      })
      .then(this.handleResponseData)
      .finally(this.handleResponseFinally);

    return this.promise;
  };

  private handleResponseData = (data: BackendResponse) => {
    if (this.props.onChange) {
      this.props.onChange(data);
    }

    return data;
  };

  public render() {
    return (
      <AttachFiles
        {...this.props}
        {...this.state}
        onRemove={this.handleOnRemove}
        onDrop={this.handleOnDrop}
        onFilesUpdate={this.updateFiles}
      />
    );
  }
}
