import * as React from 'react';
import ButtonGroup, { Props as ButtonGroupProps } from './ButtonGroup';
import { Item, Items } from './types';

type Props = ButtonGroupProps;

interface State {
  items: Items;
}

const More: Item = {
  val: 'more',
  text: 'Другие статусы',
  items: [],
};

class ButtonGroupController extends React.Component<ButtonGroupProps, State> {
  private ref = React.createRef<HTMLSpanElement>();

  private width?: {
    more: number;
    buttons: number[];
    buttonsTotal: number;
  };

  public componentDidMount() {
    this.calculateWidths();
    this.refresh();
    window.addEventListener('resize', this.refresh);
  }

  public componentDidUpdate(prevProps: Props) {
    if (!this.width) {
      this.calculateWidths();
      this.forceUpdate();
    } else if (this.props.items !== prevProps.items) {
      delete this.width;
      this.forceUpdate();
    }
  }

  public componentWillUnmount() {
    window.removeEventListener('resize', this.refresh);
  }

  private getRenderItems(): Items {
    if (!this.width) {
      return this.props.items.concat([More]);
    }

    const total = this.calculateTotal();
    const { buttonsTotal, more, buttons } = this.width;

    if (total >= buttonsTotal) {
      return this.props.items;
    }

    // find optimal render
    let lastRenderButtonIndex = 0;
    let renderButtonsWidth = more + buttons[0];

    while (renderButtonsWidth < total) {
      lastRenderButtonIndex += 1;
      renderButtonsWidth += buttons[lastRenderButtonIndex];
    }

    const moreButtonScheme: Item = { ...More };
    moreButtonScheme.items = this.props.items.slice(lastRenderButtonIndex);

    return this.props.items.slice(0, lastRenderButtonIndex).concat([moreButtonScheme]);
  }

  private refresh = () => {
    this.forceUpdate();
  };

  private calculateWidths() {
    if (this.ref.current) {
      const childrenWidth = Array.from(this.ref.current.children).map(
        (elem) => elem.getBoundingClientRect().width,
      );

      this.width = {
        more: childrenWidth[childrenWidth.length - 1],
        buttons: childrenWidth.slice(0, -1),
        buttonsTotal: childrenWidth.slice(0, -1).reduce((a, b) => a + b, 0),
      };
    }
  }

  private calculateTotal(): number {
    if (this.ref.current) {
      return this.ref.current.getBoundingClientRect().width;
    }

    return 0;
  }

  public render() {
    const { items, ...props } = this.props;

    return (
      <ButtonGroup items={this.getRenderItems()} {...props} getWrapRef={this.ref} width="max" />
    );
  }
}

type PropsDecorator = Omit<Props, 'items'> & { items?: Items };

// fix items === undefined throw error on render
export default (props: PropsDecorator) => {
  const { items } = props;

  if (!(Array.isArray(items) && items.length)) {
    return null;
  }

  return React.createElement(ButtonGroupController, props as Props);
};
