import React, { MouseEvent, ReactNode, ComponentType, RefObject } from 'react';
import createI18N from '@yandex-int/i18n';
import Button, { ButtonViewProp } from '@crm/components/dist/lego2/Button';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import cx from 'classnames';
import { TooltipWithDelay } from '@crm/components/dist/lego2/Tooltip';
import { copyToClipboard } from 'utils/copyToClipboard';
import { IconProvider } from '@yandex-lego/components/esm/next/Button/Icon/Button-Icon';
import { Direction } from '@crm/components/dist/lego2/withOldDirections';
import Icon from 'lego/components/Icon';
import * as keyset from './CopyButton/CopyButton.i18n';

interface Component {
  onClick: (event: MouseEvent) => void;
}

interface State {
  popupVisible: boolean;
}

interface Props {
  copy: string | number;
  children?: ReactNode;
  component?: ComponentType<Component>;
  className?: string;
  iconProvider?: IconProvider;
  iconClassName?: string;
  stopPropagation?: boolean;
  view?: ButtonViewProp;
  size?: LegoSizeProp;
  title?: string;
  withTooltip?: boolean;
  tooltipText?: string;
  tooltipDirection?: Direction | Direction[];
  tooltipCloseDelay?: number;
}

const i18n = createI18N(keyset);
const linkCopied = i18n('linkCopied');

class CopyButton extends React.Component<Props, State> {
  state = {
    popupVisible: false,
  };
  buttonRef: RefObject<HTMLElement> = React.createRef();

  private closeTooltip = () => this.setState({ popupVisible: false });

  private handleClick = (event: MouseEvent) => {
    const { copy } = this.props;

    if (this.props.stopPropagation) {
      event.stopPropagation();
    }

    if (this.props.withTooltip) {
      this.setState({ popupVisible: true });
    }

    copyToClipboard(copy as string);
  };

  private iconProvider = (iconCls) => {
    const { iconClassName, iconProvider } = this.props;
    if (iconProvider) {
      return iconProvider(cx(iconCls, iconClassName));
    }

    return (
      <Icon className={cx(iconCls, iconClassName)} pack={Icon.PACKS.FONTAWESOME} icon="copy" />
    );
  };

  public render() {
    const {
      copy,
      iconClassName,
      component: Component = Button,
      children,
      withTooltip,
      tooltipText = linkCopied,
      tooltipDirection = 'right-center' as Direction,
      ...other
    } = this.props;

    return (
      <>
        {Component === Button && children == null ? (
          <Component
            ref={this.buttonRef}
            icon={this.iconProvider}
            {...other}
            onClick={this.handleClick}
          />
        ) : (
          <Component {...other} onClick={this.handleClick} />
        )}
        {withTooltip ? (
          <TooltipWithDelay
            anchor={this.buttonRef}
            visible={this.state.popupVisible}
            state="success"
            direction={tooltipDirection}
            size="s"
            view="default"
            padding="xs"
            hasTail
            onDelayEnd={this.closeTooltip}
          >
            {tooltipText}
          </TooltipWithDelay>
        ) : null}
      </>
    );
  }
}

export default CopyButton;
