import React from 'react';
import PropTypes from 'prop-types';
import Textinput from '@crm/components/dist/lego2/Textinput';
import 'moment/locale/ru';
import { DateField, Calendar } from 'jk-react-date-picker';
import 'jk-react-date-picker/index.css';
import 'styles/component/datepicker.css';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import toLocalISOString from 'utils/date/toLocalISOString';
import { dateFormat as dateFormatFunc } from 'requestForms/utils/dateConvertor';
import Toolbar from 'components/Toolbar';
import normalizeNumber from './normalizeNumber';
import css from './styles.modules.scss';

export default class DateTimePicker extends React.Component {
  constructor(props) {
    super(props);
    this.state = {
      calendarStyle: {},
    };
  }

  componentDidMount() {
    if (typeof this.props.onMount === 'function') {
      this.props.onMount();
    }

    if (!this.props.value && this.props.initOnMount) {
      // TODO: разобраться более детально с проблемой
      // в final-form не корректно отрабатывает onChange в componentDidMount контрола
      window.setTimeout(() => {
        this.setDate(this.props.initOffset);
      }, 0);
    }
  }

  componentDidUpdate(prevProps) {
    if (this.props.value === null || this.props.value === '') return;

    const prevValue = dateFormatFunc(prevProps.value);
    const newValue = dateFormatFunc(this.props.value);

    if (prevValue !== newValue) {
      this.field.setValue(newValue);
    }
  }

  onChange = (date) => {
    this.props.onChange(date ? toLocalISOString(date) : null);
  };

  setDate = (dateShift) => {
    const { days, hours } = dateShift || {};
    const date = new Date();

    if (days) {
      date.setDate(date.getDate() + days);
    }

    if (hours) {
      date.setHours(
        date.getHours() + hours,
        date.getMinutes(),
        date.getSeconds(),
        date.getMilliseconds(),
      );
    }

    if (!this.props.time) {
      date.setHours(0, 0, 0, 0);
    } else {
      date.setSeconds(0, 0);
    }

    this.onChange(date);
  };

  handleChangeDate = (_, value) => {
    const date = value && value.dateMoment && value.dateMoment.toDate();
    if (date) {
      const currentDate = new Date(this.props.value);
      if (!Number.isNaN(Number(currentDate))) {
        date.setHours(currentDate.getHours());
        date.setMinutes(currentDate.getMinutes());
      }
      if (date.getTime() !== currentDate.getTime()) this.onChange(date);
    }

    if (this.isSupportNullValue(date) && date === null) {
      this.onChange(null);
    }
  };

  handleChangeMinutes = (value) => {
    if (this.props.nullable && !this.props.value) {
      return;
    }

    const newValue = normalizeNumber(value, 0, 59);

    const date = new Date(this.props.value);
    date.setMinutes(newValue);
    this.onChange(date);
  };

  handleChangeHours = (value) => {
    if (this.props.nullable && !this.props.value) {
      return;
    }

    const newValue = normalizeNumber(value, 0, 23);

    const date = new Date(this.props.value);
    date.setHours(newValue);
    this.onChange(date);
  };

  handleFastClick = (e) => {
    this.setDate({
      days: Number(e.target.dataset.offsetDays),
      hours: Number(e.target.dataset.offsetHours),
    });
  };

  handleReset = () => {
    this.props.onChange(null);
    this.props.onReset();
  };

  renderInput = (props) => {
    const { placeholder, size, id, autoFocus, hasClear } = this.props;

    this.inputProps = props;

    return (
      <Textinput
        value={props.value}
        placeholder={placeholder}
        id={id}
        size={size}
        autoFocus={autoFocus}
        hasClear={hasClear}
        {...props}
        className={css.b__textinput}
        ref={this.getInputRef}
      />
    );
  };

  getInputRef = (node) => {
    this.inputProps.ref(node);
    this.input = node;
    this.setState({
      calendarStyle: this.calendarStyle(),
    });
  };

  getDateFieldRef = (node) => {
    this.field = node;
  };

  // dirty fix for https://st.yandex-team.ru/CRM-12316
  isSupportNullValue() {
    return !this.props.time || this.props.nullable;
  }

  enoughSpaceToShowCalendarUnderInput() {
    return this.input && window.innerHeight - this.input.getBoundingClientRect().bottom > 260;
  }

  calendarStyle() {
    if (!this.enoughSpaceToShowCalendarUnderInput()) {
      return {
        marginTop: '-280px',
      };
    }

    return {};
  }

  render() {
    const {
      className,
      minDate,
      maxDate,
      size,
      view,
      dateFormat,
      value,
      time,
      fastValues,
      hasReset,
    } = this.props;
    const date = new Date(value);

    let formattedValue = this.isSupportNullValue() && value === null ? null : dateFormatFunc(value);

    return (
      <span className={cx(className, css.b)}>
        <div className={css.b__main}>
          <DateField
            className={css.date}
            expanded={view === 'inline' ? true : undefined}
            locale="ru"
            minDate={minDate}
            maxDate={maxDate}
            dateFormat={dateFormat}
            theme={view}
            renderInput={this.renderInput}
            forceValidDate
            updateOnDateClick
            expandOnFocus
            collapseOnDateClick
            defaultValue={formattedValue}
            ref={this.getDateFieldRef}
            onChange={this.handleChangeDate}
          >
            <Calendar
              style={this.state.calendarStyle}
              forceValidDate
              highlightWeekends
              highlightToday
              weekNumbers={false}
              weekStartDay={1}
              footer={false}
              defaultValue={formattedValue}
            />
          </DateField>
          {time && (
            <span className={css.time}>
              <Textinput
                className={css.time__field}
                type="number"
                min="0"
                max="24"
                size={size}
                maxLength={2}
                onChange={this.handleChangeHours}
                value={`0${date.getHours()}`.slice(-2)}
                hasClear={false}
              />
              <span className={css.time__delimiter}>:</span>
              <Textinput
                className={css.time__field}
                type="number"
                min="0"
                max="60"
                size={size}
                maxLength={2}
                onChange={this.handleChangeMinutes}
                value={`0${date.getMinutes()}`.slice(-2)}
                hasClear={false}
              />
            </span>
          )}
        </div>
        {fastValues && (
          <Toolbar>
            <Button view="pseudo" data-offset-hours={4} onClick={this.handleFastClick}>
              Через 4 часа
            </Button>
            <Button view="pseudo" data-offset-days={1} onClick={this.handleFastClick}>
              Через 1 день
            </Button>
            <Button view="pseudo" data-offset-days={2} onClick={this.handleFastClick}>
              Через 2 дня
            </Button>
            <Button view="pseudo" data-offset-days={7} onClick={this.handleFastClick}>
              Через 7 дней
            </Button>
          </Toolbar>
        )}
        {hasReset && (
          <div className={css.clearRow}>
            <Button onClick={this.handleReset} view="danger">
              Сбросить
            </Button>
          </div>
        )}
      </span>
    );
  }
}

DateTimePicker.propTypes = {
  onMount: PropTypes.func,
  onChange: PropTypes.func,
  onReset: PropTypes.func,
  minDate: PropTypes.instanceOf(Date),
  maxDate: PropTypes.instanceOf(Date),
  value: PropTypes.string,
  size: PropTypes.string,
  dateFormat: PropTypes.string,
  view: PropTypes.string,
  placeholder: PropTypes.string,
  id: PropTypes.string,
  time: PropTypes.bool,
  autoFocus: PropTypes.bool,
  fastValues: PropTypes.bool,
  initOnMount: PropTypes.bool,
  hasClear: PropTypes.bool,
  className: PropTypes.string,
  initOffset: PropTypes.shape({
    days: PropTypes.number,
    hours: PropTypes.number,
  }),
};

DateTimePicker.defaultProps = {
  onMount() {},
  onChange() {},
  onReset() {},
  minDate: undefined,
  maxDate: undefined,
  value: undefined,
  className: undefined,
  size: 'm',
  dateFormat: 'DD.MM.YYYY',
  view: 'islands',
  placeholder: undefined,
  id: undefined,
  time: false,
  initOnMount: false,
  autoFocus: false,
  fastValues: false,
  hasClear: false,
  hasReset: false,
  initOffset: undefined,
};
