import React from 'react';
import { withStateHandlers } from 'recompose';
import debounce from 'lodash/debounce';
import { Cancelable } from 'lodash';
import { InfiniteListView } from 'containers/InfiniteList';
import Empty from 'components/Empty';
import FilteredMenu from './FilteredMenu';

const FilteredMenuWithState = withStateHandlers(
  { search: '' },
  {
    onSearch: (_, props) => value => {
      props.onSearch(value);
      return { search: value };
    },
  },
)(FilteredMenu);

const empty = <Empty theme={Empty.THEMES.EMPTY_DATA} />;

interface Props {
  name: string;
  provider: any;
  hasSearchSupport?: boolean;
  onChange: (id: number, item: any) => void;
}

interface State {
  provider: any;
}

class FilteredMenuContainer extends React.Component<Props, State> {
  public static defaultProps = {
    name: '',
    hasSearchSupport: false,
  };

  private setProvider: ((text: string) => void) & Partial<Cancelable>;

  public constructor(props) {
    super(props);

    const setProvider = text => {
      if (text) {
        this.setState({ provider: args => this.props.provider({ ...args, text }) });
      } else {
        this.setState({ provider: this.props.provider });
      }
    };

    if (props.delay) {
      this.setProvider = debounce(setProvider, props.delay);
    } else {
      this.setProvider = setProvider;
    }

    this.state = {
      provider: props.provider,
    };
  }

  public componentWillUnmount() {
    if (this.setProvider.cancel) {
      this.setProvider.cancel();
    }
  }

  private renderItems = ({ items }) => {
    const { provider, name, hasSearchSupport, ...forwardProps } = this.props;

    return hasSearchSupport ? (
      <FilteredMenuWithState items={items} {...forwardProps} onSearch={this.setProvider} />
    ) : (
      <FilteredMenu items={items} {...forwardProps} />
    );
  };

  public render() {
    const { name } = this.props;
    const { provider } = this.state;

    return (
      <InfiniteListView
        onLoad={provider}
        loadOnMount
        reloadOnLoadUpdate
        withNavigation={false}
        name={`filteredMenu/${name}`}
        emptyComponent={empty}
        inject
      >
        {this.renderItems}
      </InfiniteListView>
    );
  }
}

export default FilteredMenuContainer;
