import React, { Component } from 'react';
import PropTypes from 'prop-types';
import Popup from '@crm/components/dist/lego2/Popup';
import Button from '@crm/components/dist/lego2/Button';
import Avatar from 'components/Avatar';
import cx from 'classnames';
import { Bubble } from 'components/Bubble';
import { Text } from '@crm/components/dist/lego2/Text';
import StaffCard from 'components/StaffCard';
import Icon from 'lego/components/Icon';
import Tree, { getSlugId } from './connectorUsers';
import css from './Grants.modules.scss';

const canItemClick = (node) => Boolean(node.type);

export default class Grants extends Component {
  static TYPE_KEYS = {
    0: 'groupId',
    1: 'userId',
  };

  constructor(props) {
    super(props);
    this.state = {
      show: false,
    };
    this.node = React.createRef();
  }

  componentDidMount() {
    const {
      isNew,
      user: { id, name },
    } = this.props;

    if (isNew) {
      const userGrand = { type: 1, id, name };
      if (!this.hasGrand(userGrand)) {
        this.addGrant(userGrand);
      }
    }
  }

  getGrants = () => {
    const { value, isUsersOnly } = this.props;

    const result = [];
    if (Array.isArray(value)) {
      value.forEach((grant) => {
        const id = (isUsersOnly && grant.id) || grant.userId || grant.groupId;
        const type = isUsersOnly || grant.userId ? 1 : 0;

        result.push(getSlugId({ id, type }));
      });
    }

    return result;
  };

  togglePopup = () => {
    const { show } = this.state;
    this.setState({ show: !show });
  };

  hidePopup = () => {
    this.setState({ show: false });
  };

  toggleGrant = (id, item) => {
    const { isUsersOnly } = this.props;
    let value = this.props.value || [];
    let removedGrant;
    value = value.filter((grant) => {
      const has =
        (isUsersOnly && item.id === grant.id) ||
        (item.type === 1 && item.id === grant.userId) ||
        (item.type === 0 && item.id === grant.groupId);

      if (has) {
        removedGrant = grant;
      }

      return !has;
    });

    if (removedGrant) {
      this.props.onChange(value, { type: 'remove', item: removedGrant });
    } else {
      this.addGrant(item);
    }
  };

  hasGrand = (item) => {
    if (!item) {
      return false;
    }

    const { value } = this.props;

    return (value || []).some((grand) => {
      if (grand.groupId || grand.userId) {
        return item.id === grand[Grants.TYPE_KEYS[item.type]];
      }

      return item.id === grand.id;
    });
  };

  addGrant = (item) => {
    const { value } = this.props;

    const result = {};
    if (item.type === 0) {
      result.groupId = item.id;
      result.groupName = item.name;
    }
    if (item.type === 1) {
      result.userId = item.id;
      result.userName = item.name;
    }

    const action = {
      type: 'add',
      item,
    };

    this.props.onChange((value || []).concat(result), action);
  };

  addCurrentUser = () => {
    this.addGrant({ ...this.props.user, type: 1 });
  };

  removeGrant = (item) => {
    const { value } = this.props;

    const key = item.userId ? 'userId' : 'groupId';
    const keyValue = item[key];

    let newValue = (value || []).filter((i) => i[key] !== keyValue);

    if (!newValue.length) {
      newValue = null;
    }

    const action = {
      type: 'remove',
      item,
    };

    this.props.onChange(newValue, action);
  };

  clearGrants = () => {
    this.props.onChange(null, { type: 'clear' });
  };

  render() {
    const { className, value, user, staffcard, staffcardTheme, disabled, isUsersOnly } = this.props;

    const hasCurrentUser = this.hasGrand({ ...user, type: 1 });

    const tree = (
      <Tree
        selected={this.getGrants()}
        onClick={this.toggleGrant}
        onClear={this.clearGrants}
        canItemClick={isUsersOnly ? canItemClick : undefined}
        destroyOnUnmount
      />
    );

    return (
      <div className={cx(css.root, className)}>
        {!disabled && (
          <span className={hasCurrentUser ? undefined : css.b__addWrap}>
            <Button
              view="pseudo"
              className={css.b__add}
              pin="circle-circle"
              size="xs"
              innerRef={this.node}
              onClick={this.togglePopup}
              title="Добавить пользователя"
              icon={(iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="add" />}
            />
            {!hasCurrentUser && (
              <Button
                view="clear"
                className={css.b__user}
                size="xs"
                onClick={this.addCurrentUser}
                title="Добавить меня"
                icon={(iconCls) => (
                  <span>
                    <Avatar user={user} size="xs" theme="round" className={css.b__userAvatar} />
                    <Icon
                      pack={Icon.PACKS.MATERIAL}
                      icon="add"
                      size="xs"
                      className={cx(css.b__userPlus, iconCls)}
                    />
                  </span>
                )}
              />
            )}
            <Popup
              visible={this.state.show}
              onOutsideClick={this.hidePopup}
              directions="bottom"
              autoZIndex
              padding="xs"
              target="anchor"
              anchor={this.node}
            >
              {tree}
            </Popup>
          </span>
        )}
        <span className={css.tags}>
          {Array.isArray(value) &&
            value.map((item) => {
              let name;
              let key;
              let showStaffCard = null;
              if (item.userId) {
                if (staffcard) {
                  showStaffCard = true;
                }

                name = item.userName;
                key = `userId${item.userId}`;
              } else if (item.groupId) {
                name = item.groupName;
                key = `groupId${item.groupId}`;
              } else {
                if (staffcard) {
                  showStaffCard = true;
                }

                ({ name } = item);
                key = item.id;
              }

              const tagContent = showStaffCard ? (
                <StaffCard theme={staffcardTheme} user={item} size="xs" />
              ) : (
                <Text typography="control-m">{name}</Text>
              );

              return (
                <Bubble
                  key={key}
                  className={cx(css.bubble, css.tag)}
                  canDelete={!disabled}
                  onDelete={() => {
                    this.removeGrant(item);
                  }}
                  view={staffcardTheme === 'avatar' ? 'clear' : undefined}
                >
                  {tagContent}
                </Bubble>
              );
            })}
        </span>
      </div>
    );
  }
}

Grants.propTypes = {
  className: PropTypes.string,
  value: PropTypes.oneOfType([PropTypes.instanceOf(Array), PropTypes.instanceOf(Object)]),
  isNew: PropTypes.bool,
  user: PropTypes.shape({
    id: PropTypes.number,
    name: PropTypes.string,
  }).isRequired,
  onChange: PropTypes.func,
  disabled: PropTypes.bool,
  staffcard: PropTypes.bool,
  staffcardTheme: PropTypes.string,
  isUsersOnly: PropTypes.bool,
};

Grants.defaultProps = {
  className: undefined,
  value: null,
  isNew: false,
  disabled: false,
  staffcard: false,
  staffcardTheme: StaffCard.THEME.YANDEX,
  onChange: () => {},
  isUsersOnly: false,
};
