import * as React from 'react';
import cx from 'classnames';
import Textinput from '@crm/components/dist/lego2/Textinput';
import Popup from '@crm/components/dist/lego2/Popup';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import Menu from '@crm/components/dist/lego2/Menu';
import { MixedItem } from '@yandex-lego/components/Menu';
import { ITextinputProps } from '@yandex-lego/components/Textinput/desktop/bundle';
import { MenuProps } from './SuggestBase.types';
import './Suggest.css';
import css from './Suggest.scss';
import { GetKey, ItemComponent } from './types';

export interface Props {
  className?: string;
  items?: MixedItem[];
  disabledMap?: { [key: string]: boolean };
  size?: LegoSizeProp;
  pin?: ITextinputProps['pin'];
  text?: string;
  placeholder?: string;
  hasClear?: boolean;
  autoFocus?: boolean;
  focused?: boolean;
  value?: string[];
  getKey: GetKey;
  component: ItemComponent;
  onOutsideClick?: () => void;
  onChangeText?: (text: string) => void;
  onChangeMenu?: (value: string | string[]) => void;
  onBlur?: React.FocusEventHandler<HTMLElement>;
  onFocus?: React.FocusEventHandler<HTMLElement>;
  addonAfterTextInput?: React.ReactNode;
}

class SuggestBase extends React.Component<Props> {
  public static defaultProps = {
    hasClear: true,
  };

  public static MENU_ALLOW_KEYS = ['ArrowUp', 'ArrowDown', 'Enter'];

  public textinput: React.RefObject<HTMLSpanElement> = React.createRef();

  private _keysToMenu: boolean = false;

  protected textinputRow: React.RefObject<HTMLSpanElement> = React.createRef();

  public hasItems = (): boolean => {
    const { items } = this.props;

    return Array.isArray(items) && Boolean(items.length);
  };

  private handleKeyDown = (e: React.KeyboardEvent<HTMLInputElement>): void => {
    const { key } = e;

    if (key === 'Enter') {
      // block submit form by enter
      e.preventDefault();
    }

    // https://st.yandex-team.ru/CRM-10278
    if (key === ' ') {
      e.nativeEvent.stopImmediatePropagation();
    }

    const isShowPopup = this.hasItems();
    const isMenuAllowKey = SuggestBase.MENU_ALLOW_KEYS.includes(key);

    if (isMenuAllowKey) {
      if (isShowPopup) {
        this._keysToMenu = true;
      } else {
        e.preventDefault();
        this._keysToMenu = false;
      }
    }
  };

  private handleKeyUp = (): void => {
    const { items } = this.props;
    const isShowPopup = items && Array.isArray(items);

    if (isShowPopup && this._keysToMenu) {
      this._keysToMenu = false;
    }
  };

  private onMenuChange = (event) => {
    const { onChangeMenu } = this.props;
    if (onChangeMenu) {
      const values =
        (Array.isArray(event.target.value) && event.target.value.filter((value) => value)) ||
        event.target.value;
      onChangeMenu(values);
    }
  };

  public renderInput(className: string = ''): React.ReactNode {
    const {
      hasClear,
      onChangeText,
      onFocus,
      onBlur,
      text,
      autoFocus,
      size,
      placeholder,
      pin,
      addonAfterTextInput,
    } = this.props;

    return (
      <span className={cx(className, css.b__input)} ref={this.textinputRow}>
        <Textinput
          ref={this.textinput}
          onFocus={onFocus}
          onBlur={onBlur}
          value={text}
          onChange={onChangeText}
          hasClear={hasClear}
          onKeyDown={this.handleKeyDown}
          onKeyUp={this.handleKeyUp}
          autoFocus={autoFocus}
          size={size}
          placeholder={placeholder}
          pin={pin}
        />
        {addonAfterTextInput && <span>{addonAfterTextInput}</span>}
      </span>
    );
  }

  public renderMenu(menuProps: MenuProps = {}): React.ReactNode {
    if (!this.hasItems()) {
      return null;
    }

    const { items, value, getKey, component: Component, disabledMap, focused } = this.props;
    const { style } = menuProps;

    return (
      <Menu
        className={cx(css.b__list, 'SuggestBase-List')}
        style={style}
        onChange={this.onMenuChange}
        value={value}
        width="max"
        focused={focused}
        items={(items || []).map((item) => {
          const val = getKey(item);
          return {
            value: val,
            disabled: disabledMap && disabledMap[val],
            content: <Component item={item} />,
          };
        })}
      />
    );
  }

  public render(): React.ReactFragment {
    const { onOutsideClick } = this.props;

    const isShowPopup = this.hasItems();

    return (
      <React.Fragment>
        {this.renderInput()}
        {isShowPopup && (
          <Popup
            visible
            target="anchor"
            anchor={this.textinput}
            onOutsideClick={onOutsideClick}
            directions="bottom"
          >
            {this.renderMenu()}
          </Popup>
        )}
      </React.Fragment>
    );
  }
}

export default SuggestBase;
