import React from 'react';
import Promise from 'bluebird';
import isPromise from 'is-promise';
import memoize from 'memoize-one';
import { WithOId } from 'types';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import { ITextinputProps } from '@yandex-lego/components/Textinput/desktop/bundle';
import Suggest from './Suggest';
import Provider, { ProviderTypes } from '../utils/Provider';
import { GetKey, ItemComponent, View } from './types';

export interface Props extends Partial<WithOId> {
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  value: any;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  onChange: (value: any) => void;
  className?: string;
  getKey: GetKey;
  autoFocus?: boolean;
  showOnFocus?: boolean;
  isCustomValue?: boolean; // значение может быть текст введенный в поисковую строку
  provider: ProviderTypes;
  placeholder?: string;
  size?: LegoSizeProp;
  pin?: ITextinputProps['pin'];
  component: ItemComponent;
  view?: View;
  addonAfterTextInput?: React.ReactNode;
  focused?: boolean;
}

interface State {
  text: string;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  items: any[] | null;
}

class SuggestContainer extends React.Component<Props, State> {
  public static View = View;

  public static defaultProps = {
    getKey: (k) => k,
  };

  public state = {
    text: '',
    items: [],
  };

  private provider = new Provider();

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  private promise: Promise<any> | null;

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  private itemsMap: { [key: string]: any };

  public componentWillUnmount() {
    if (this.promise) {
      this.promise.cancel();
      this.promise = null;
    }
  }

  private getItems = (text: string): void => {
    const { provider } = this.props;

    if (this.promise) {
      this.promise.cancel();
      this.promise = null;
    }

    const result = this.provider.parse(provider)(text, this.props.oId);

    if (isPromise(result)) {
      this.promise = result;
      result
        .then((data) => {
          this.promise = null;
          this.setItems(data.items);
        })
        .catch((e) => {
          this.promise = null;
          throw e;
        });
    } else if (Array.isArray(result)) {
      this.setItems(result);
    }
  };

  private setItems = (items) => {
    const { getKey } = this.props;

    const formattedItems = Array.isArray(items) ? items : [];

    this.itemsMap = formattedItems.reduce((ac, item) => {
      // eslint-disable-next-line no-param-reassign
      ac[getKey(item)] = item;

      return ac;
    }, {});

    this.setState({ items: formattedItems });
  };

  private getDisabledMap = memoize((value) => {
    const { getKey } = this.props;

    return (value || []).reduce((ac, c) => {
      // eslint-disable-next-line no-param-reassign
      ac[getKey(c)] = c.disabled;

      return ac;
    }, {});
  });

  private handleChangeText = (text: string): void => {
    this.setState({ text });
    this.getItems(text);
  };

  private handleChangeMenu = (value: string | string[]): void => {
    this.props.onChange(this.formatValue(value));
  };

  private handleFocus = () => {
    if (this.props.showOnFocus || this.state.text) {
      this.getItems(this.state.text);
    }
  };

  private handleBlur = () => {
    const { isCustomValue, value, onChange } = this.props;
    if (isCustomValue && this.state.text && !(value && value.length)) {
      onChange([{ name: this.state.text }]);
    }
  };

  private handleOutsideClick = () => {
    this.setState({ items: null });
  };

  private formatValue = (value) => {
    const { getKey, value: oldValue } = this.props;

    const resultItemsMap = (oldValue || []).reduce((ac, c) => {
      // eslint-disable-next-line no-param-reassign
      ac[getKey(c)] = c;
      return ac;
    }, {});

    return (value || []).map((key) => {
      const item = this.itemsMap[key] || resultItemsMap[key];
      resultItemsMap[key] = item;

      return item;
    });
  };

  private parseValue = (value) => {
    const { getKey } = this.props;

    return (value || []).map((item) => getKey(item));
  };

  public render(): React.ReactElement {
    const {
      value,
      getKey,
      component,
      autoFocus,
      size,
      pin,
      placeholder,
      view,
      className,
      addonAfterTextInput,
      focused,
    } = this.props;
    const { items, text } = this.state;

    return (
      <Suggest
        className={className}
        value={this.parseValue(value)}
        text={text}
        disabledMap={this.getDisabledMap(value)}
        items={items}
        onFocus={this.handleFocus}
        onBlur={this.handleBlur}
        onChangeText={this.handleChangeText}
        onChangeMenu={this.handleChangeMenu}
        onOutsideClick={this.handleOutsideClick}
        getKey={getKey}
        component={component}
        autoFocus={autoFocus}
        size={size}
        pin={pin}
        placeholder={placeholder}
        view={view}
        addonAfterTextInput={addonAfterTextInput}
        focused={focused}
      />
    );
  }
}

export default SuggestContainer;
