import React from 'react';
import Button from '@crm/components/dist/lego2/Button';
import Popup from '@crm/components/dist/lego2/Popup';
import Icon from '@crm/components/dist/lego2/Icon';
import get from 'lodash/get';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import { Direction } from '@crm/components/dist/lego2/withOldDirections';
import { ITextinputProps } from '@yandex-lego/components/Textinput/desktop/bundle';
import SuggestWithTags from './SuggestWithTags';
import { ProviderTypes } from '../utils/Provider';
import { ItemComponent } from './types';
import css from './SuggestInPopup.module.css';

const spanStyles = {
  minWidth: 0,
  flex: 1,
  overflow: 'hidden',
  textOverflow: 'ellipsis',
};

const DIRECTIONS: Direction[] = ['bottom-center', 'bottom-left', 'bottom-right'];

const style = { width: '100%', minWidth: 0, flexGrow: 1, flexBasis: 0 };

interface State {
  visible: boolean;
}

interface Props {
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  value: any;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  onChange: (value: any) => void;
  provider: ProviderTypes;
  size: LegoSizeProp;
  pin?: ITextinputProps['pin'];
  defaultValueText: string;
  displayKeySlug: string;
  width?: 'max' | 'auto';
  labelWidth?: number;
  labelPrefix: React.ReactNode;
  component?: ItemComponent;
  isSingleValue?: boolean;
  isNullEmptyValue?: boolean;
  hideSuggestIfSingleValue?: boolean;
  addonAfterTextInput?: React.ReactNode;
}

export default class SuggestInPopup extends React.Component<Props, State> {
  private button: React.RefObject<HTMLElement> = React.createRef();

  public static defaultProps = {
    size: 'l',
    defaultValueText: '',
    displayKeySlug: 'name',
    labelPrefix: null,
  };

  /*
   * Баг - закрытие попапа по клику на "удалить тег":
   * 1) удаляем тег в попапе (жмем крестик)
   * 2) тег удаляется из DOM
   * 3) попап проверяет должен ли он закрыться: проверяет popup.contains(e.target);
   * 4) так как тег уже не в DOM попап закрывается
   *
   * Фикс:
   * Откладываем onChange. Получается, что проверка на contains происходит первая
   * */
  private onChangeTimeout: number;

  public constructor(props) {
    super(props);

    this.state = { visible: false };
  }

  public componentWillUnmount(): void {
    window.clearTimeout(this.onChangeTimeout);
  }

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  private handleChange = (value: any) => {
    this.onChangeTimeout = window.setTimeout(() => {
      this.props.onChange(value);
    });
  };

  private show = () => {
    this.setState({ visible: true });
  };

  private hide = () => {
    this.setState({ visible: false });
  };

  private toggle = () => {
    if (this.state.visible) {
      this.hide();
    } else {
      this.show();
    }
  };

  public render() {
    const {
      displayKeySlug,
      value,
      provider,
      size,
      pin,
      labelPrefix,
      labelWidth,
      defaultValueText,
      width,
      component,
      isSingleValue,
      isNullEmptyValue,
      hideSuggestIfSingleValue,
      addonAfterTextInput,
    } = this.props;

    const selectedFolderName = get(value, `[0].${displayKeySlug}`);
    const count = get(value, 'length');

    return (
      <React.Fragment>
        <Button
          style={style}
          size={size}
          onClick={this.toggle}
          ref={this.button}
          width={width}
          iconRight={(iconCls) => <Icon className={iconCls} glyph="carets-v" />}
        >
          {labelPrefix && (
            <span style={{ color: 'grey', width: labelWidth, textAlign: 'left', float: 'left' }}>
              {labelPrefix}
            </span>
          )}
          <span style={spanStyles}>
            {(count > 1 && `Выбрано ${count}`) || selectedFolderName || defaultValueText}
          </span>
        </Button>
        <Popup
          zIndex={1000}
          onOutsideClick={this.hide}
          target="anchor"
          anchor={this.button}
          directions={DIRECTIONS}
          visible={this.state.visible}
          padding="xs"
        >
          <SuggestWithTags
            className={css.b__suggest}
            size={size}
            pin={pin}
            view={SuggestWithTags.View.Inline}
            value={value}
            onChange={this.handleChange}
            provider={provider}
            component={component}
            autoFocus
            isSingleValue={isSingleValue}
            isNullEmptyValue={isNullEmptyValue}
            hideSuggestIfSingleValue={hideSuggestIfSingleValue}
            addonAfterTextInput={addonAfterTextInput}
            focused={this.state.visible}
          />
        </Popup>
      </React.Fragment>
    );
  }
}
