import * as React from 'react';
import { WithOId } from 'types';
import Promise from 'bluebird';
import { baseCallApi } from 'api/common';
import Button from '@crm/components/dist/lego2/Button';
import createI18N from '@yandex-int/i18n';
import memoize from 'memoize-one';
import ArrayValueManager from 'utils/ArrayValueManager';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import * as keyset from './SuggestStaff.i18n';
import SuggestUserItem from './SuggestUserItem';
import SuggestUserGroupItem from './SuggestUserGroupItem';

const i18nMe = createI18N(keyset)('me');

enum StaffSuggestVariant {
  All = 'All',
  UsersOnly = 'UsersOnly',
  GroupsOnly = 'GroupsOnly',
  UsersInGroupOnly = 'UsersInGroupOnly',
}

export enum EType {
  User = 'User',
  UserGroup = 'UserGroup',
}

export interface User {
  id: number;
  name: string;
  // eslint-disable-next-line
  [key: string]: any;
}

interface UserGroup {
  id: number;
  url: string;
  name: string;
}

type SyntheticObj<T> = { [P in keyof T]: T[P] } & {
  syntheticId: string;
  eType: EType;
};

export type SyntheticUser = SyntheticObj<User>;
export type SyntheticUserGroup = SyntheticObj<UserGroup>;

export type SyntheticAny = SyntheticUser | SyntheticUserGroup;

export const isSyntheticUser = (obj: SyntheticAny): obj is SyntheticUser => {
  return obj.eType === EType.User;
};

export const isSyntheticUserGroup = (obj: SyntheticAny): obj is SyntheticUserGroup => {
  return obj.eType === EType.UserGroup;
};

interface StaffOId {
  eId: number;
  eType: EType;
}

interface Response {
  map: {
    users: { [key: string]: User[] };
    userGroups: { [key: string]: UserGroup[] };
  };
  items: StaffOId[];
  eof: boolean;
}

interface Props extends Partial<WithOId> {
  // eslint-disable-next-line
  value: any;
  // eslint-disable-next-line
  onChange: (value: any) => void;
  variant?: StaffSuggestVariant;
  onlyUserHadAnyRole?: boolean;
  onlyUserActive?: boolean;
  onlyUserNotDismissed?: boolean;
  size?: LegoSizeProp;
  user?: User;
  isSingleValue?: boolean;
  isNullEmptyValue?: boolean;
  module?: string;
}

const TYPE_TO_MAP_KEY = {
  User: 'users',
  UserGroup: 'userGroups',
};

const TYPE_TO_COMPONENT = {
  User: SuggestUserItem,
  UserGroup: SuggestUserGroupItem,
};

interface ItemProps {
  // eslint-disable-next-line
  item: any;
  eType: string;
}

const Item = (props: ItemProps) => {
  const { item } = props;
  const { eType } = item;

  const Component = TYPE_TO_COMPONENT[eType];

  if (!Component) {
    return null;
  }

  return <Component {...props} />;
};

export const createSyntheticObj = (eType: EType, item: User | UserGroup) => ({
  ...item,
  syntheticId: `${item.id}:${eType}`,
  eType,
});

const createSyntheticUser = (user: User) => createSyntheticObj(EType.User, user);

class SuggestStaffBase2 extends React.Component<Props> {
  public static StaffSuggestVariant = StaffSuggestVariant;

  private formatValue = memoize((value) => {
    const { isSingleValue } = this.props;

    return ArrayValueManager.valueToArray(value, isSingleValue).map((item) => {
      if (item && item.syntheticId) {
        return item;
      }

      return createSyntheticUser(item as User);
    });
  });

  private formatUser = memoize(createSyntheticUser);

  // eslint-disable-next-line
  protected component: React.ComponentType<any> = () => null;

  // eslint-disable-next-line
  private getSuggest = (text?: string): Promise<any> | any[] => {
    const {
      variant,
      onlyUserHadAnyRole,
      onlyUserActive,
      onlyUserNotDismissed,
      module,
      oId,
    } = this.props;

    if (!text) {
      return [];
    }

    return Promise.delay(650).then(() => {
      const promise = baseCallApi({
        url: '/suggest/staff',
        data: {
          text,
          variant,
          onlyUserHadAnyRole,
          onlyUserActive,
          onlyUserNotDismissed,
          module,
          ...oId,
        },
        global: false,
      }) as Promise<Response>;

      return promise.then((data: Response) => {
        return {
          items: data.items.reduce((ac: SyntheticAny[], staffOId: StaffOId) => {
            const item = data.map[TYPE_TO_MAP_KEY[staffOId.eType]][staffOId.eId];
            if (item) {
              ac.push(createSyntheticObj(staffOId.eType, item));
            }

            return ac;
          }, []),
          eof: data.eof,
        };
      });
    });
  };

  private getKey = (item) => item.syntheticId;

  private handleAddUser = (): void => {
    const { user, value, onChange, isSingleValue, isNullEmptyValue } = this.props;

    if (!user) {
      return;
    }

    const arrayValueManager = new ArrayValueManager(this.formatValue(value));

    if (arrayValueManager.has(user)) {
      return;
    }

    arrayValueManager.add(this.formatUser(user));

    onChange(
      ArrayValueManager.arrayToValue(arrayValueManager.getValue(), isSingleValue, isNullEmptyValue),
    );
  };

  public render(): React.ReactElement {
    const {
      variant,
      onlyUserHadAnyRole,
      onlyUserActive,
      onlyUserNotDismissed,
      size,
      user,
      value,
      ...other
    } = this.props;

    const Component = this.component;

    const buttonAddMe = user && (
      <Button view="pseudo" size={size} onClick={this.handleAddUser} pin="clear-round">
        {i18nMe}
      </Button>
    );

    return (
      <Component
        component={Item}
        provider={this.getSuggest}
        getKey={this.getKey}
        addonAfterTextInput={buttonAddMe}
        size={size}
        pin="round-brick"
        value={this.formatValue(value)}
        {...other}
      />
    );
  }
}

export default SuggestStaffBase2;
