import React from 'react';
import themes from '../themes';
import { TabProps, TabElement } from '../Tab';
import { Value, ThemeType } from '../types';
import { TabsProps, TabsState } from './Tabs.types';
import { TabsView } from '../TabsView';
import { TabToggle } from '../TabToggle';

export class Tabs extends React.Component<TabsProps, TabsState> {
  public static defaultProps = {
    navigationTheme: 'center',
    canHide: false,
    showHeader: true,
    defaultValue: 0,
    theme: ThemeType.Default,
    tone: 'default',
  };

  public static getChildIndex(props: TabsProps, state: TabsState): Value {
    return props.onChange ? props.value || '' : state.value;
  }

  private cacheLazyMountMap: { [key: string]: boolean } = {};

  private cacheChildrenMap: { [key: string]: TabElement } = {};

  private cacheChildrenArray: React.ReactElement<TabProps>[] | null = null;

  public constructor(props) {
    super(props);

    this.state = {
      value: props.defaultValue,
    };
  }

  public componentDidUpdate(prevProps, prevState) {
    const { onDidUpdate } = this.props;
    const prevTabIndex = Tabs.getChildIndex(prevProps, prevState) as number;
    const tabIndex = Tabs.getChildIndex(this.props, this.state) as number;

    if (typeof onDidUpdate === 'function' && prevTabIndex !== tabIndex) {
      onDidUpdate(prevTabIndex, tabIndex);
    }
  }

  private handleTabClick = (value) => () => {
    let formattedValue = value;

    if (this.props.onChange) {
      if (this.props.canHide && this.props.value === formattedValue) {
        formattedValue = -1;
      }
      this.props.onChange(formattedValue);
    } else {
      if (this.props.canHide && this.state.value === formattedValue) {
        formattedValue = -1;
      }
      this.setState({ value: formattedValue });
    }
  };

  private isCurrentIndex(index): boolean {
    return Tabs.getChildIndex(this.props, this.state) === index;
  }

  private renderTabs(): React.ReactElement | null {
    const { theme, width } = this.props;

    const childrenWithProps = React.Children.map(this.cacheChildrenArray, (child, i) => {
      if (React.isValidElement(child)) {
        const { keepMount, lazyMount, value } = child.props;
        const id = value || i;
        const isCurrentTab = this.isCurrentIndex(id);
        if (
          (keepMount && (!lazyMount || (lazyMount && this.cacheLazyMountMap[id]))) ||
          isCurrentTab
        ) {
          this.cacheLazyMountMap[id] = true;

          return React.cloneElement(child, {
            hidden: !isCurrentTab,
            theme,
            width,
          });
        }
      }

      return null;
    });

    return <>{childrenWithProps}</>;
  }

  public render(): React.ReactElement {
    const {
      className,
      classNameHeader,
      style,
      navigationTheme,
      title,
      theme,
      headerLeft,
      headerRight,
      content,
      width,
      tone,
      showHeader,
      testid
    } = this.props;

    const cssTheme = themes[theme] || {};

    const childIndex = Tabs.getChildIndex(this.props, this.state);

    this.cacheChildrenMap = {};
    this.cacheChildrenArray = React.Children.map<TabElement, TabElement>(
      this.props.children,
      (child, index) => {
        const valueMerge = child.props.value || index;
        const themeMerge = child.props.theme || theme;

        const clonedChild = React.cloneElement<TabProps>(child, {
          value: valueMerge,
          theme: themeMerge,
        });
        this.cacheChildrenMap[valueMerge] = clonedChild;

        return clonedChild;
      },
    );

    const buttons = React.Children.map(this.cacheChildrenArray, (child) => {
      if (React.isValidElement(child)) {
        const { value: val, title, buttonAfterContent, buttonBeforeContent, testid } = child.props;

        return (
          <TabToggle
            theme={cssTheme}
            active={childIndex === val}
            onClick={this.handleTabClick(val)}
            beforeContent={buttonBeforeContent}
            afterContent={buttonAfterContent}
            navigationTheme={navigationTheme}
            testid={testid}
          >
            {title}
          </TabToggle>
        );
      }

      return null;
    });

    return (
      <TabsView
        theme={cssTheme}
        tone={tone}
        width={width}
        showHeader={showHeader}
        title={title}
        style={style}
        className={className}
        classNameHeader={classNameHeader}
        headerLeft={headerLeft}
        headerRight={headerRight}
        buttons={buttons}
        testid={testid}
      >
        {content}
        {this.renderTabs()}
      </TabsView>
    );
  }
}
