/* eslint-disable @typescript-eslint/no-explicit-any */
import React, { CSSProperties } from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { cnTheme } from '@yandex-lego/components/Theme';
import { crmTheme } from '@crm/components/dist/lego2/Theme/presets/themes';
import { TagSize } from './types';
import css from './Tag.module.css';
import { processColor } from './processColor';

export interface Props {
  className?: string;
  partClassNames?: { button: string; label: string };
  color?: string;
  onDelete?: (event: React.MouseEvent<HTMLButtonElement>) => void;
  onClick?: (event: React.MouseEvent<HTMLButtonElement>) => void;
  render?: any;
  children?: React.ReactNode;
  size?: TagSize;
  theme?: string;
  tagRef?: React.Ref<HTMLSpanElement>;
  item?: any;
  title?: string;
  disabled?: boolean;
}

const Tag: React.FC<Props> = (props) => {
  const {
    className,
    color = '#ffffff',
    partClassNames = { button: '', label: '' },
    size = 'xs',
    onClick,
    onDelete,
    children,
    tagRef,
    render,
    theme,
    item,
    title,
    disabled = false,
  } = props;

  const colorObj = processColor(color);

  let lastDeleteTimeStamp = -1;

  const handleClick = (event) => {
    if (!onClick || lastDeleteTimeStamp === event.timeStamp) {
      return;
    }

    onClick(event);
  };

  const handleDelete = (event) => {
    if (!onDelete) {
      return;
    }

    lastDeleteTimeStamp = event.timeStamp;
    onDelete(event);
  };

  let { isShowBorder } = colorObj;
  const bgStyle: CSSProperties = {};
  const colorStyle: CSSProperties = {};
  if (theme !== 'clear') {
    bgStyle.background = colorObj.bgColor;
    colorStyle.color = colorObj.textColor;
  }

  if (disabled) {
    bgStyle.background = '#e9e9e9';
    colorStyle.color = '#7a7a7a';
    isShowBorder = false;
  }

  return (
    <span
      className={cx(
        className,
        css.b,
        css[`b_size_${size}`],
        css[`b_theme_${theme}`],
        cnTheme({ root: crmTheme.root }),
        {
          [css.b_border]: isShowBorder,
          [css.b_disabled]: disabled,
        },
      )}
      onClick={onClick ? handleClick : undefined}
      role={onClick ? 'button' : undefined}
      style={bgStyle}
      ref={tagRef}
      title={title}
    >
      <span className={cx(partClassNames.label, css.b__text)} style={colorStyle}>
        {render ? render(item, props) : children}
      </span>
      {!disabled && onDelete && (
        <Button
          size={size as any}
          view="clear"
          className={cx(partClassNames.button, css.b__remove)}
          style={colorStyle}
          onClick={handleDelete}
          icon={(cls) => <Icon size={size as any} glyph="type-cross" className={cls} />}
          aria-label="remove"
        />
      )}
    </span>
  );
};

export default Tag;
