import React from 'react';
import PropTypes from 'prop-types';
import Tooltip from '@crm/components/dist/lego2/Tooltip/Tooltip';
import cx from 'classnames';
import { DIRECTIONS } from '@crm/components/dist/lego2/Popup/Popup.config';
import css from './ToolTip.module.css';

class ToolTip extends React.Component {
  static propTypes = {
    children: PropTypes.node,
    tooltip: PropTypes.node,
    showDelay: PropTypes.number,
    hideDelay: PropTypes.number,
    className: PropTypes.string,
    title: PropTypes.string,
  };

  static defaultProps = {
    children: undefined,
    tooltip: undefined,
    className: undefined,
    title: undefined,
    hideDelay: 0,
    showDelay: 0,
  };

  ref = React.createRef();

  constructor(props) {
    super(props);

    this.state = {
      visible: false,
    };

    this.hideWithDelay = this.withDelay('hide');
    this.showWithDelay = this.withDelay('show');
  }

  componentWillUnmount() {
    this.clearTimer();
  }

  withDelay = (methodName) => {
    const delayKey = `${methodName}Delay`;

    return () => {
      this.clearTimer();
      if (this.props[delayKey]) {
        this.timer = setTimeout(() => {
          this[methodName]();
        }, this.props[delayKey]);
      } else {
        this[methodName]();
      }
    };
  };

  show = () => {
    this.setState({ visible: true });
  };

  hide = () => {
    this.setState({ visible: false });
  };

  clearTimer() {
    clearTimeout(this.timer);
  }

  handleClick = (event) => {
    if (this.props.onClick) {
      this.props.onClick(event);
    }

    if (this.props.stopPropagation) {
      event.stopPropagation();
    }
  };

  render() {
    const {
      children,
      tooltip,
      hideDelay,
      className,
      title,
      tooltipClassName,
      hasTail = true,
      ...other
    } = this.props;
    const bodyRef = { current: document.body };

    return (
      <span
        role="presentation"
        ref={this.ref}
        onMouseEnter={this.showWithDelay}
        onMouseLeave={this.hideWithDelay}
        className={className}
        title={title}
      >
        {children}
        <Tooltip
          hasTail={hasTail}
          tailSize={12}
          visible={this.state.visible && Boolean(tooltip)}
          direction={DIRECTIONS}
          anchor={this.ref}
          keepMounted={false}
          boundary={bodyRef}
          size={other.padding}
          view="default"
          className={cx(css.ToolTip, tooltipClassName)}
          {...other}
          onClick={this.handleClick}
        >
          {tooltip}
        </Tooltip>
      </span>
    );
  }
}

export default ToolTip;
