import React, { Suspense } from 'react';
import Button from '@crm/components/dist/lego2/Button';
import WikiFormatter from 'components/WikiFormatter';
import cx from 'classnames';
import { Toolbar2 } from 'components/Toolbar2';
import { RollerWithPopup } from 'components/Roller';
import WFInput from './WFInput';
import css from './WithPreview.module.css';

class WFInputWithPreview extends React.Component {
  static defaultProps = {
    view: 'default',
    actionsPosition: 'left',
  };

  state = { isPreview: false };

  refInput = React.createRef();
  containerRef = React.createRef();

  componentDidUpdate(prevProps) {
    const { value } = this.props;

    if (!value && value !== prevProps.value) {
      // если для комментария нет текста выходим из режима превью
      // например это решает проблему: при сохраненнии комментария выходить из режима превью,
      // так как в этом случае мы очищаем поле
      this.closePreviewMode();
    }
  }

  openPreviewMode = () => {
    this.setState({ isPreview: true });
  };

  closePreviewMode = () => {
    this.setState({ isPreview: false });
  };

  reply = (text) => {
    if (this.refInput.current) {
      this.refInput.current.reply(text);
    }
  };

  render() {
    const {
      actions,
      children,
      className,
      addonToolbarBefore,
      addonToolbarAfter,
      addonToolbarRightAfter,
      view,
      actionsPosition,
      ...other
    } = this.props;

    return (
      <div ref={this.containerRef} className={cx(className, css[`b_view_${view}`])}>
        {this.state.isPreview && (
          <Suspense fallback={<div />}>
            <div className={css.b__previewHeader}>
              You are viewing the unsaved version of this comment.
            </div>
            <div className={css.b__content}>
              <WikiFormatter markdown={this.props.value} />
            </div>
          </Suspense>
        )}
        <WFInput hidden={this.state.isPreview} ref={this.refInput} view={view} {...other} />
        {children}
        {this.renderToolbar()}
      </div>
    );
  }

  renderToolbar() {
    const { actions } = this.props;

    const actionsWithPreview = (
      <>
        {actions}
        {this.state.isPreview ? (
          <Button onClick={this.closePreviewMode}>Редактировать</Button>
        ) : (
          <Button onClick={this.openPreviewMode} disabled={!this.props.value}>
            Просмотреть
          </Button>
        )}
      </>
    );

    return (
      <Toolbar2
        className={cx(css.b__toolbar, css.b__toolbar_flexRow)}
        left={this.renderToolbarLeft(actionsWithPreview)}
        right={this.renderToolbarRight(actionsWithPreview)}
      />
    );
  }

  renderToolbarLeft(actions) {
    const { addonToolbarBefore, addonToolbarAfter, actionsPosition } = this.props;

    return (
      <>
        {addonToolbarBefore}
        {actionsPosition === 'left' && actions}
        {addonToolbarAfter}
      </>
    );
  }

  renderToolbarRight(actions) {
    const { addonToolbarRightAfter, actionsPosition } = this.props;

    return (
      <>
        <RollerWithPopup
          className={css.b__toolbar__FooterRoller}
          rollerClassNames={{
            containerClassName: css.b__toolbar__FooterRollerContainer,
          }}
          popupProps={{
            containerClassName: css.b__toolbar__FooterRollerPopupContainer,
            scope: 'inplace',
            boundary: this.containerRef,
          }}
        >
          {actionsPosition === 'right' && actions}
        </RollerWithPopup>
        {addonToolbarRightAfter}
      </>
    );
  }
}

export default WFInputWithPreview;
