import React, { Component, ReactNode } from 'react';
import { get } from 'api/common';

const cache = new Map();

export interface DictionaryProps {
  endpoint: string;
  useCache: boolean;

  /**
   * Какой список из response body брать, если ручка грузит сразу несколько списков.
   * По умолчанию - items.
   */
  loadedListKey?: string;
}

interface DictionaryState {
  items?: { id: number | string; name: string }[];
}

interface WrappedComponentProps {
  disabled?: boolean;
  options?: { value: number | string; content: ReactNode }[];
}

const withDictionary = <T extends WrappedComponentProps>(config?: Partial<DictionaryProps>) => (
  WrappedComponent: React.ComponentType<T>,
) => {
  class Dictionary extends Component<DictionaryProps & T, DictionaryState> {
    public static defaultProps = {
      useCache: true,
      loadedListKey: 'items',
      ...config,
    };

    public constructor(props) {
      super(props);

      const { endpoint, useCache } = this.props;

      this.state = {
        items: useCache ? cache.get(endpoint) : undefined,
      };
    }

    public componentDidMount() {
      const { endpoint, useCache, loadedListKey } = this.props;
      const { items } = this.state;

      if (!items) {
        get({ url: endpoint }).then((data) => {
          const backendItems = data[loadedListKey];

          if (useCache) {
            cache.set(endpoint, backendItems);
          }

          this.setState({ items: backendItems });
        });
      }
    }

    public render() {
      const { items } = this.state;

      /*
       * Используем key, чтобы при загрузке сработал componentDidMount у Select с isAutoSelectFirst
       * */
      let key = 'loaded';
      const props: WrappedComponentProps = {};
      if (!items) {
        props.disabled = true;
        props.options = [];
        key = 'loading';
      } else {
        props.options = items.map((item) => ({
          value: String(item.id),
          content: item.name,
        }));
      }

      return <WrappedComponent key={key} {...this.props} {...props} />;
    }
  }

  return Dictionary;
};

export default withDictionary;
