import React from 'react';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import { FieldRenderProps } from 'react-final-form';
import { FormContext } from 'components/FinalForm/Form/Form.context';
import getDisplayName from 'utils/getDisplayName';

interface FinalFormFieldExtension<Value> {
  size?: LegoSizeProp;
  isCheckBox?: boolean;
  hasClear?: boolean;
  isTouchOnChange?: boolean;
  destroyValueOnUnmount?: boolean;
  finalFormPropsMap?: (
    props: FieldRenderProps<Value>,
  ) => {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    [key: string]: any;
  };
}

export type WithFinalFormFieldProps<Value, P> = FieldRenderProps<Value> &
  FinalFormFieldExtension<Value> &
  P & { inputProps?: Partial<P> };

// eslint-disable-next-line @typescript-eslint/no-explicit-any
const withFinalFormField = <P extends {}, Value = any>(
  config?: Partial<P> & FinalFormFieldExtension<Value>,
) => (WrappedComponent) => {
  class WithFormikField extends React.Component<WithFinalFormFieldProps<Value, P>> {
    static contextType = FormContext;
    public context: React.ContextType<typeof FormContext>;

    public static defaultProps = {
      size: 's' as LegoSizeProp | undefined,
      hasClear: true,
      isCheckBox: false,
      ...config,
    };

    public static displayName = `WithFinalFormField(${getDisplayName(WrappedComponent)})`;

    public componentWillUnmount(): void {
      if (this.props.destroyValueOnUnmount) {
        this.handleChange(null);
      }
    }

    private handleChange = (value) => {
      const { input, isTouchOnChange } = this.props;

      this.context.onChange({ value, name: input.name });
      input.onChange(value);

      if (isTouchOnChange) {
        input.onBlur();
      }
    };

    public render() {
      const {
        input,
        meta,
        isCheckBox,
        isTouchOnChange,
        destroyValueOnUnmount,
        inputProps,
        refComponent,
        finalFormPropsMap = () => ({}),
        ...other
      } = this.props;

      const passProps = {
        ...input,
        ...other,
        onChange: this.handleChange,
        ...inputProps,
        ...finalFormPropsMap({ input, meta }),
      };

      if (isCheckBox && input) {
        passProps.checked = Boolean(input.value);
      }

      return <WrappedComponent ref={refComponent} {...passProps} />;
    }
  }

  return WithFormikField;
};

export default withFinalFormField;
