import React from 'react';
import { FieldProps } from 'formik';
import getDisplayName from 'utils/getDisplayName';

interface FormikFieldExtension {
  isObjectValue?: boolean;
  isCheckBox?: boolean;
  isTouchOnChange?: boolean;
  passBlurHandle?: boolean;
  hasClear?: boolean;
}

export type WithFormikFieldProps<P> = FieldProps &
  FormikFieldExtension &
  P & { inputProps?: Partial<P> };

export type WithFormikFieldComponent<P> = React.ComponentClass<WithFormikFieldProps<P>>;

const withFormikField = <P extends {}>(config: Partial<P> & FormikFieldExtension) => (
  WrappedComponent,
) => {
  class WithFormikField extends React.Component<WithFormikFieldProps<P>> {
    public static defaultProps = {
      size: 's',
      hasClear: true,
      isObjectValue: false,
      isCheckBox: false,
      isTouchOnChange: false,
      passBlurHandle: true,
      ...config,
    };

    public static displayName = `WithFormikField(${getDisplayName(WrappedComponent)})`;

    private handleChange = (value) => {
      const { field, form, isTouchOnChange } = this.props;

      form.setFieldValue(field.name, value);

      if (isTouchOnChange) {
        window.setTimeout(() => {
          form.setFieldTouched(field.name);
        });
      }
    };

    public render() {
      const {
        field,
        form,
        isObjectValue,
        isCheckBox,
        inputProps,
        passBlurHandle,
        ...other
      } = this.props;

      const passProps = { ...field, ...other, ...inputProps };

      if (isObjectValue) {
        passProps.onChange = this.handleChange;
      }

      if (isCheckBox && field) {
        passProps.checked = Boolean(field.value);
      }

      if (!passBlurHandle) {
        delete (passProps as Partial<typeof passProps>).onBlur;
      }

      return <WrappedComponent {...passProps} />;
    }
  }

  return WithFormikField;
};

export default withFormikField;
