import React from 'react';
import { makeObservable, observable, action } from 'mobx';
import { ModalService } from 'types/ModalService';
import { ModalContainer } from './ModalContainer';
import { ModalProps, FormComponentType, ModalFormOptions, ModalFormProps } from './ModalForm.types';

import { ReactModalForm } from './ReactModalForm';

export class ModalForm<P extends ModalFormProps> implements ModalService<P> {
  private options: ModalFormOptions<P>;
  public isMinimized: boolean = false;
  public isVisible: boolean = true;
  public name: string = '';

  public constructor(
    keyOrOptions: string | ModalFormOptions<P>,
    FormClass: FormComponentType<P> = () => null,
    modalProps?: ModalProps,
    formProps?: Partial<P>,
  ) {
    if (typeof keyOrOptions === 'string') {
      this.options = {
        key: keyOrOptions,
        formComponent: FormClass,
        modalProps,
        formProps,
      };
    } else {
      this.options = keyOrOptions;
    }
    makeObservable(this, {
      isMinimized: observable,
      isVisible: observable,
      name: observable,
      minimize: action.bound,
      maximize: action.bound,
      setVisible: action.bound,
      setName: action.bound,
    });
  }

  public minimize = () => {
    this.isMinimized = true;
  };

  public maximize = () => {
    this.isMinimized = false;
  };

  public setVisible = (visible: boolean) => {
    this.isVisible = visible;
  };

  public setName = (name: string) => {
    this.name = name;
  };

  public open = (props: P, modalProps?: Partial<ModalProps>) => {
    if (this.options.onLoad) {
      this.options
        .onLoad(props)
        .then((response) => {
          this.openModal(props, modalProps, response);
          return response;
        })
        .catch(() => {
          this.destroy();
        });
    } else {
      this.openModal(props, modalProps);
    }
  };

  private openModal = (props: P, modalProps?: Partial<ModalProps>, data?: unknown) => {
    if (ModalContainer.instance) {
      const modalForm = (
        <ReactModalForm<P>
          formProps={{ ...this.options.formProps, ...props, initialData: data }}
          modalProps={{ ...this.options.modalProps, ...modalProps }}
          onClose={this.close}
          formComponent={this.options.formComponent}
          modalService={this}
        />
      );

      ModalContainer.instance.setContent(this.options.key, modalForm);
      this.setVisible(true);
    }
  };

  public destroy = (keepMounted: boolean = false) => {
    if (ModalContainer.instance) {
      this.maximize();
      this.setName('');
      this.setVisible(false);
      if (!keepMounted) {
        ModalContainer.instance.removeContentByKey(this.options.key);
      }
    }
  };

  public close = () => {
    this.destroy(this.options.modalProps?.keepMounted);
  };
}
