import { schema, normalize } from 'normalizr';
import { camelizeKeys } from 'humps';
import { baseCallApi } from 'api/common';

// Fetches an API response and normalizes the result JSON according to schema.
// This makes every API response have the same shape, regardless of how nested it was.
// eslint-disable-next-line no-shadow,max-params
const callApi = (endpoint, callArgs, schema, params, dataDecorator, provider) => {
  let promise;
  if (typeof provider === 'function') {
    promise = provider();
  } else {
    promise = baseCallApi({
      url: endpoint,
      type: 'get',
      global: true,
      isHandleUnsuccessful: true,
      ...callArgs,
    });
  }

  return promise.then((_data) => {
    let data = _data;
    if (typeof dataDecorator === 'function') {
      data = dataDecorator(data);
    }

    const camelizedJson = camelizeKeys(data);
    const rest = {};
    if (params && params.dataKey) {
      rest[params.dataKey] = data[params.dataKey];
    }
    if (schema) {
      return Object.assign({}, normalize(camelizedJson, schema), { data: rest });
    }

    return data;
  });
};

// We use this Normalizr schemas to transform API responses from a nested form
// to a flat form where repos and users are placed in `entities`, and nested
// JSON objects are replaced with their IDs. This is very convenient for
// consumption by reducers, because we can easily build a normalized tree
// and keep it updated as we fetch more data.

// Read more about Normalizr: https://github.com/paularmstrong/normalizr

// GitHub's API may return results with uppercase letters while the query
// doesn't contain any. For example, "someuser" could result in "SomeUser"
// leading to a frozen UI as it wouldn't find "someuser" in the entities.
// That's why we're forcing lower cases down there.
const queueItemSchema = new schema.Entity('queueItemSchema');

const generateSlug = () => 'panel';

const queuePanelSchema = new schema.Entity('queuePanelSchema', {}, { idAttribute: generateSlug });

queuePanelSchema.define({
  items: [queueItemSchema],
});

queueItemSchema.define({
  items: [queueItemSchema],
});

// Schemas for Github API responses.
export const Schemas = {
  QUEUE: queueItemSchema,
  QUEUE_ARRAY: [queueItemSchema],
  QUEUE_PANEL: queuePanelSchema,
};

// Action key that carries API call info interpreted by this Redux middleware.
export const CALL_API = 'CALL_API';

// A Redux middleware that interprets actions with CALL_API info specified.
// Performs the call and promises when such actions are dispatched.
export default (store) => (next) => (action) => {
  const callAPI = action[CALL_API];

  if (typeof callAPI === 'undefined') {
    return next(action);
  }

  let { endpoint } = callAPI;

  const {
    schema, // eslint-disable-line no-shadow
    types,
    params,
    callArgs,
    dataDecorator,
    provider,
    callbacks,
  } = callAPI;

  if (typeof endpoint === 'function') {
    endpoint = endpoint(store.getState());
  }

  if (typeof endpoint !== 'string') {
    throw new Error('Specify a string endpoint URL.');
  }

  if (!Array.isArray(types) || types.length !== 3) {
    throw new Error('Expected an array of three action types.');
  }

  if (!types.every((type) => typeof type === 'string')) {
    throw new Error('Expected action types to be strings.');
  }

  const actionWith = (data) => {
    const finalAction = Object.assign({}, action, data);
    delete finalAction[CALL_API];
    return finalAction;
  };

  const [requestType, successType, failureType] = types;
  next(actionWith({ type: requestType, meta: action.meta }));
  return callApi(endpoint, callArgs, schema, params, dataDecorator, provider).then(
    (response) => {
      if (callbacks && callbacks.success) {
        callbacks.success(response, store.dispatch);
      }

      return next(
        actionWith({
          response,
          type: successType,
          meta: action.meta,
        }),
      );
    },
    (error) => {
      const message = error.message;
      if (callbacks && callbacks.error) {
        callbacks.error(error, store.dispatch);
      }
      return next(
        actionWith({
          type: failureType,
          error: message || 'Something bad happened',
          meta: action.meta,
        }),
      );
    },
  );
};
