import { baseCallApi, jsonApiCall } from 'api/common';
import isPromise from 'is-promise';
import backendUpdate from 'middlewares/backendUpdate';

const middleware = store => next => action => {
  if (action.type !== 'FETCH') {
    return next(action);
  }

  let context = null;
  if (action.context) {
    if (action.context.get) {
      context = action.context.get(store.getState());
    } else {
      context = backendUpdate.getContextArgs(action.context, store.getState());
    }
  }

  const [REQUEST, RECEIVE, FAIL] = action.actions || [];

  // action before request
  if (REQUEST) {
    store.dispatch({ type: REQUEST, meta: action.meta });
  }

  let api;

  if (action.json || action.context) {
    api = jsonApiCall;
  } else {
    api = baseCallApi;
  }

  let promise = api({ ...action.args, data: { ...action.args.data, context } });

  if (typeof action.promiseCallback === 'function') {
    const result = action.promiseCallback(promise);
    if (isPromise(result)) {
      promise = result;
    }
  }

  return promise
    .then(data => {
      // context handle
      if (action.context) {
        if (action.context.update) {
          store.dispatch(action.context.update(data));
        } else {
          backendUpdate.update(action.context, data.context, store.dispatch, store.getState());
        }
      }

      // return promise data if no action type
      if (!RECEIVE) {
        return data;
      }

      // or return result of dispatch action
      return store.dispatch({ type: RECEIVE, data, args: action.args, meta: action.meta });
    })
    .catch(error => {
      if (FAIL) {
        store.dispatch({ type: FAIL, error, meta: action.meta });
      }

      throw error;
    });
};

export default middleware;
