import { logger } from 'services/Logger';
import actions from 'actions';

export const LOGGER_ACTION_TYPE = 'CRASH_REPORT';

/**
 * @param func - callback
 * @param count - count before block
 * @param blockDelay - ms block delay
 * @param seqRequestInterval - ms interval for request in same sequence
 * @returns function
 * */
const customThrottle = (func, count = 10, blockDelay = 20000, seqRequestInterval = 1000) => {
  let lastRequestTime = 0;
  let seqCount = 0;

  return (...args) => {
    const currentRequestTime = Number(new Date());

    if (
      (currentRequestTime - lastRequestTime > seqRequestInterval && seqCount < count) ||
      currentRequestTime - lastRequestTime > blockDelay
    ) {
      seqCount = 0;
    }

    if (seqCount < count) {
      lastRequestTime = currentRequestTime;
      seqCount += 1;
      func(...args);
    }
  };
};

const log = (store, action) => {
  const { errorObj, alertText, alertType, errorLevel, isLogOnBackend = true } = action;

  if (isLogOnBackend) {
    if (errorObj) {
      logger.reportError(errorObj, errorLevel);
    }
  }

  if (!(errorObj && errorObj.isShowed)) {
    store.dispatch(actions.uiToggleSnackbar(true, alertText, alertType));
  }
};

const throttleLog = customThrottle(log);

export default (store) => (next) => (action) => {
  if (action.type !== LOGGER_ACTION_TYPE) {
    try {
      return next(action);
    } catch (error) {
      logger.reportError(error);
      throw error;
    }
  }

  throttleLog(store, action);

  return null;
};
