/* eslint-disable react/jsx-no-bind */
import React, { useCallback, useEffect, useState, useRef } from 'react';
import cn from 'classnames';
import Worker from 'workers/search.worker';
import { GroupItem, MessageType } from 'workers/search.types';
import { TemplateList } from 'modules/adminPanel/modules/templates/types';
import { FixedSizeList } from 'react-window';
import AutoSizer from 'react-virtualized-auto-sizer';
import css from './List.scss';

interface Props {
  className?: string;
  items?: TemplateList[];
  selectedId?: number | null;
  url: string;
  groups?: boolean;
  search?: string;
}

const List: React.FC<Props> = ({ items = [], search = '', className, url, selectedId, groups }) => {
  const [foundItems, setFoundItems] = useState<GroupItem[]>([]);
  const workerRef = useRef<Worker>();

  useEffect(() => {
    const worker = new Worker();
    const handler = ({ data }) => setFoundItems(data);
    worker.addEventListener('message', handler);
    workerRef.current = worker;
    return () => {
      worker.removeEventListener('message', handler);
      worker.terminate();
    };
  }, []);

  useEffect(() => {
    workerRef.current?.postMessage({ type: MessageType.SEARCH, data: search });
  }, [search]);

  useEffect(() => {
    workerRef.current?.postMessage({ type: MessageType.LOAD, data: items as GroupItem[] });
  }, [items]);

  const renderItem = useCallback(
    ({ index, style }) => {
      const item = foundItems[index];
      const active = selectedId === item.id;
      const name = item.name || 'Без названия';

      return item.type === 'group' ? (
        <div className={cn(css.b_group)} style={style}>
          {name}
        </div>
      ) : (
        <a
          className={cn(css.b_item, { [css.b_active]: active })}
          href={`#${url}/${item.id}`}
          data-template-id={item.id}
          style={style}
        >
          {item.canRead === false && <i className={cn('fa', 'fa-ban', css.b_restricted)} />}
          {item.target && <span className={css.b_target}>{item.target}</span>}
          {name}
        </a>
      );
    },
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [foundItems, selectedId],
  );

  return (
    <div className={cn(className, { [css.b_groups]: groups })}>
      <AutoSizer>
        {({ width, height }) => (
          <FixedSizeList itemCount={foundItems.length} width={width} height={height} itemSize={20}>
            {renderItem}
          </FixedSizeList>
        )}
      </AutoSizer>
    </div>
  );
};

export default List;
