import React, { FC, useEffect, useRef, CSSProperties } from 'react';
import { autorun } from 'mobx';
import { useDispatch } from 'react-redux';
import { observer } from 'mobx-react-lite';
import { change } from 'redux-form';
import constants from '../../../constants';
import { Row } from './Row';
import { CategoryRightsProps } from './CategoryRights.types';
import css from './CategoryRights.module.css';
import { ValuesStore } from './ValuesStore';

export const CategoryRights: FC<CategoryRightsProps> = observer((props) => {
  const {
    input: { value, name },
    meta,
  } = props;
  const dispatch = useDispatch();
  const valuesStoreRef = useRef(new ValuesStore());
  const valuesStore = valuesStoreRef.current;

  useEffect(() => {
    if (!meta.initial) {
      return;
    }

    valuesStore.setInitialValue(meta.initial.values);
  }, [meta.initial, valuesStore]);

  useEffect(
    () =>
      autorun(() => {
        const { values } = valuesStore;
        dispatch(
          change(constants.name, name, {
            ...value,
            values,
          }),
        );
      }),
    [valuesStore, value, dispatch, name],
  );

  if (!value || !value.columns || !value.rows) {
    return null;
  }

  const { columns = [], rows = [] } = value;

  const style: CSSProperties = {
    gridTemplateColumns: `repeat(${columns.length + 1}, 1fr)`,
  };

  return (
    <div className={css.CategoryRights}>
      <div className={css.CategoryRights__table} style={style}>
        <div />
        {columns.map((column) => (
          <div className={css.CategoryRights__headerCell} key={column.id}>
            {column.name}
          </div>
        ))}

        {rows.map((row) => (
          <Row
            key={row.id}
            id={row.id}
            name={`${row.name} (${row.id})`}
            columns={columns}
            values={valuesStore.values[row.id]}
            onChange={valuesStore.change}
          />
        ))}
      </div>
    </div>
  );
});
