import React from 'react';
import { reduxForm } from 'redux-form';
import { FormattedDate } from 'react-intl';
import Prompt from 'components/Prompt';
import Row from 'forms/Base/Row';
import Button from '@crm/components/dist/lego2/Button';
import cx from 'classnames';
import { TagBubble } from 'components/TagBubble';
import StaffCard from 'components/StaffCard';
import Form from 'components/Form';
import Toolbar from 'components/Toolbar';
import { CategoryRights } from './CategoryRights';
import constants from '../../constants';
import css from './Form.scss';

const CATEGORY_TYPES = [
  { value: 'Claim', content: 'Претензия' },
  { value: 'Consultation', content: 'Консультация' },
  { value: 'Action', content: 'Действие' },
];

const TAG_FIELDS = {
  name: {
    required: true,
  },
  grants: {
    required: true,
  },
};

const TARGET_ITEMS = [
  {
    value: 'SpCall',
    content: 'Карточка звонка',
  },
  {
    value: 'Chat',
    content: 'Чат',
  },
  {
    value: 'Mail',
    content: 'Персональная почта',
  },
  {
    value: 'Account',
    content: 'Аккаунт',
  },
];

const CategoryForm = (props) => {
  const { handleSubmit, className, initialValues, pristine, submitting, invalid, onClone } = props;

  return (
    <Form className={cx(className, css.b)} onSubmit={handleSubmit}>
      <Prompt when={!pristine} />
      <div className={css.b__header}>
        <Toolbar inline>
          <Button
            view="action"
            type="submit"
            disabled={(initialValues.id && pristine) || submitting || invalid}
          >
            Сохранить
          </Button>
          {initialValues.id && (
            <Button onClick={onClone} view="pseudo">
              Клонировать
            </Button>
          )}
        </Toolbar>
        {initialValues.id && (
          <div className={css.b__meta}>
            <div className={css.b__field}>
              <span className={css.b__label}>id:</span>
              <span className={css.b__value}>{initialValues.id}</span>
              &nbsp;
            </div>
            {initialValues.modifiedOn && (
              <div className={css.b__field}>
                <span className={css.b__label}>Последнее изменение:</span>
                <span className={css.b__value}>
                  {initialValues.modifiedBy && (
                    <React.Fragment>
                      <StaffCard user={initialValues.modifiedBy} theme={StaffCard.THEME.YANDEX} />
                      ,&nbsp;
                    </React.Fragment>
                  )}
                  <FormattedDate
                    value={new Date(initialValues.modifiedOn)}
                    year="numeric"
                    month="numeric"
                    day="numeric"
                    hour="numeric"
                    minute="numeric"
                  />
                </span>
                &nbsp;
              </div>
            )}
          </div>
        )}
      </div>
      <div className={css.b__content}>
        <Row label="Название" component="TextInput" name="name" />
        <Row
          label="Родительская категория"
          name="parent"
          component="TreeInput"
          endpoint="/view/support/category/list/admin"
        />
        <Row label="Топ" component="CheckBox" name="top" />
        <Row label="Причина" component="CheckBox" name="reason" />
        <Row label="Архивная" component="CheckBox" name="isArchived" />
        <Row
          label="Показывать для"
          component="Select"
          hasEmptyValue={false}
          name="target"
          options={TARGET_ITEMS}
        />
        <Row
          label="Тип"
          component="Select"
          name="categoryType"
          options={CATEGORY_TYPES}
          isNullEmptyValue
        />
        <Row label="Id формы заявки" component="TextInput" name="reqExtFormId" fieldWidth="auto" />
        <Row
          key={initialValues.id}
          label="Ответственные"
          component="Grants"
          name="modifyGrants"
          isNew={!initialValues.id}
        />
        <Row
          label="Ключевые слова"
          component="Tags"
          name="tags"
          type="Category"
          formFields={TAG_FIELDS}
          tagComponent={({ onDelete, id, name, className, ...props }) => (
            <TagBubble
              {...props}
              className={cx(className, css.b__tag)}
              onDelete={() => onDelete(id)}
              canDelete={Boolean(onDelete)}
              tag={{ text: name }}
            />
          )}
        />
        <Row
          label="Id категории до разделения"
          component="TextInput"
          name="originCategoryId"
          fieldWidth="auto"
        />
        <Row label="Ссылка в подсказке" component="TextInput" name="tipUrl" />
        <Row label="Ссылка на подсказку из daas" component="TextInput" name="daasTipUrl" />
        <Row label="Разделять по каналам" component="CheckBox" name="isDaasTipSplittedByChannels" />
        <Row label="Доступы к категориям" component={CategoryRights} name="categoryRights" />
      </div>
    </Form>
  );
};

CategoryForm.defaultProps = {
  meta: {},
  initialValues: {},
};

export default reduxForm({ form: constants.name })(CategoryForm);
