import React from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import { delay } from 'bluebird';
import get from 'lodash/get';
import shallowDiffers from 'utils/shallowDiffers';
import GrantsModel from 'lego/components/Grants/model';
import { adaptToSave, adaptToInitialValues } from './CategoryRights';
import Form from './Form';
import createRedux from '../../redux/form';

const GRANTS_FIELDS = ['grants', 'modifyGrants'];

class FormContainer extends React.Component {
  componentDidMount() {
    const { id, isNew, query } = this.props;
    this.load({ id, isNew, query });
  }

  componentDidUpdate(prevProps) {
    const { id, isNew, query } = this.props;

    if (
      id !== prevProps.id ||
      isNew !== prevProps.isNew ||
      shallowDiffers(query, prevProps.query)
    ) {
      this.load({ id, isNew, query });
    }
  }

  componentWillUnmount() {
    this.cancel();
  }

  cancel() {
    if (this.promise) {
      this.promise.cancel();
    }
  }

  load(data) {
    this.cancel();
    this.promise = this.props.load(data);
    this.promise.then(() => {
      this.promise = null;
      return null;
    });
  }

  onSubmit = (data) => {
    const adaptedCategoryRights = adaptToSave(data.categoryRights);
    const sendData = { ...data, categoryRights: adaptedCategoryRights, validate: true };

    GRANTS_FIELDS.forEach((name) => {
      sendData[name] = GrantsModel.formatToBackend(
        data[name],
        get(this.props, `data.data.${name}`),
      );
    });

    let promise;
    const makeRequest = (data) => {
      promise = this.props.save(data);
    };

    makeRequest(sendData);
    promise.catch(async () => {
      sendData.validate = false;

      await delay(0);

      if (window.confirm('Вы уверены?')) {
        makeRequest(sendData);
      }
    });
    return promise;
  };

  toInitialValues = (data = {}) => {
    return {
      ...data,
      target: data.target || [],
      categoryRights: adaptToInitialValues(data.categoryRights),
    };
  };

  render() {
    const { isLoad, data = {}, ...props } = this.props;

    if (isLoad) {
      return (
        <Form
          meta={data.metadata}
          onSubmit={this.onSubmit}
          initialValues={this.toInitialValues(data.data)}
          {...props}
          enableReinitialize
        />
      );
    }

    return null;
  }
}

export default connect(
  (state) => createRedux.redux.selectors.getState(state),
  (dispatch) => bindActionCreators(createRedux.redux.actions, dispatch),
)(FormContainer);
