import React, { PureComponent, ReactNode } from 'react';
import { Form } from 'react-final-form';
import differenceWith from 'lodash/differenceWith';
import isEqual from 'lodash/isEqual';
import Textinput from '@crm/components/dist/lego2/Textinput';
import Button from '@crm/components/dist/lego2/Button';
import { calculatePristineByValues } from 'modules/adminPanel/utils';
import { getQueueOptionList, createQueueIdKey } from '../utils';
import Toolbar from './Toolbar';
import QueueOptionsList from '../QueueOptionsList';
import { LayoutState } from './Layout.types';
import { FormValues, QueueOption, QueueMap } from '../types';
import { getQueueOptions } from '../requests/getQueueOptions';
import { setQueueOptions } from '../requests/setQueueOptions';
import css from './Layout.module.css';

class Layout extends PureComponent<{}, LayoutState> {
  public constructor(props: {}) {
    super(props);
    this.state = {
      searchText: '',
      queueIds: [],
      queueMap: {},
      values: {
        queueOptionMap: {},
      },
    };
  }

  public async componentDidMount(): Promise<void> {
    const { queueOptions } = await getQueueOptions();
    let list: number[] = [];
    let map: QueueMap = {};

    queueOptions.forEach((queueOption) => {
      const { queueId } = queueOption;
      list.push(queueId);
      map[createQueueIdKey(queueId)] = queueOption;
    });

    this.setState((state) => ({
      queueIds: list,
      queueMap: map,
      values: {
        ...state.values,
        queueOptionMap: map,
      },
    }));
  }

  private onSubmit = (values: FormValues): Promise<void> | void => {
    const { queueIds } = this.state;
    const prevQueueOptions = getQueueOptionList(queueIds, this.state.values.queueOptionMap);
    const queueOptions = getQueueOptionList(queueIds, values.queueOptionMap);

    const changedOptions = (differenceWith(
      queueOptions,
      prevQueueOptions,
      isEqual,
    ) as QueueOption[]).map((qo) => ({
      ...qo,
      outgoingCallCategoryRequired: false,
      outgoingMailCategoryRequired: false,
    }));

    if (changedOptions.length) {
      const promise = setQueueOptions({ queueOptions: changedOptions });
      promise.then(() => {
        this.setState({
          values,
        });
      });

      return promise;
    }
  };

  private handleSearch = (text: string): void => {
    this.setState({
      searchText: text,
    });
  };

  public render(): ReactNode {
    return (
      <div className={css.Layout}>
        <div className={css.Layout__list}>
          <Toolbar position="top">
            <Textinput
              placeholder="Поиск"
              value={this.state.searchText}
              onChange={this.handleSearch}
            />
          </Toolbar>

          <Form onSubmit={this.onSubmit} initialValues={this.state.values}>
            {(props) => {
              const pristine = calculatePristineByValues(
                this.state.values.queueOptionMap,
                props.values.queueOptionMap,
              );

              return (
                <form onSubmit={props.handleSubmit} className={css.Layout__form}>
                  <QueueOptionsList
                    queueIds={this.state.queueIds}
                    queueMap={this.state.queueMap}
                    searchText={this.state.searchText}
                  />
                  <Toolbar>
                    <Button view="action" type="submit" disabled={props.submitting || pristine}>
                      Сохранить
                    </Button>
                  </Toolbar>
                </form>
              );
            }}
          </Form>
        </div>
      </div>
    );
  }
}

export default Layout;
