import React, { PureComponent, ReactNode, CSSProperties } from 'react';
import AutoSizer from 'react-virtualized-auto-sizer';
import { get } from 'api/common';
import { FixedSizeList as List } from 'react-window';
import cx from 'classnames';
import debounce from 'lodash/debounce';
import { getQueueOptionList, createQueueIdKey } from '../utils';
import { QueueOptionsListState, QueueOptionsListProps, ItemData } from './QueueOptionsList.types';
import QueueOption from '../QueueOption';
import { QueueOption as IQueueOption } from '../types';
import css from './QueueOptionsList.module.css';

const Row = ({ data, index, style }: { data: ItemData; index: number; style: CSSProperties }) => {
  const { visibleQueueIds, communicationTypes } = data;
  const name = `queueOptionMap[${createQueueIdKey(visibleQueueIds[index])}]`;
  return (
    <QueueOption key={name} name={name} style={style} communicationTypes={communicationTypes} />
  );
};

class QueueOptionsList extends PureComponent<QueueOptionsListProps, QueueOptionsListState> {
  private queueOptions: IQueueOption[] = [];

  public constructor(props: QueueOptionsListProps) {
    super(props);
    this.state = {
      visibleQueueIds: [],
      communicationTypes: [],
    };
  }

  private setVisibleQueueIds = debounce((searchText: string): void => {
    const visibleQueueIds: number[] = [];
    this.queueOptions.forEach((queueOption) => {
      if (queueOption.queueName.toLowerCase().indexOf(searchText.toLowerCase()) !== -1) {
        visibleQueueIds.push(queueOption.queueId);
      }
    });
    this.setState({
      visibleQueueIds,
    });
  }, 200);

  public async componentDidMount() {
    const { items } = await get({
      url: '/view/dicts/communicationType',
    });

    this.setState({
      communicationTypes: items.map((item) => ({
        value: item.id,
        content: item.name,
      })),
    });
  }

  public componentDidUpdate(prevProps: QueueOptionsListProps) {
    if (prevProps.queueIds !== this.props.queueIds || prevProps.queueMap !== this.props.queueMap) {
      this.queueOptions = getQueueOptionList(this.props.queueIds, this.props.queueMap);
      this.setVisibleQueueIds(this.props.searchText);
    }

    if (prevProps.searchText !== this.props.searchText) {
      this.setVisibleQueueIds(this.props.searchText);
    }
  }

  public render(): ReactNode {
    const { visibleQueueIds } = this.state;
    const itemData: ItemData = {
      visibleQueueIds,
      communicationTypes: this.state.communicationTypes,
    };
    return (
      <div className={css.QueueOptionsList}>
        <div className={css.QueueOptionsList__Labels}>
          <div className={css.QueueOptionsList__Label}>Обязательность категоризатора</div>
          <div className={cx(css.QueueOptionsList__Label, css.QueueOptionsList__Label_visibility)}>
            Видимость категоризатора
          </div>
          <div className={css.QueueOptionsList__Label}>Коммуникация по умолчанию</div>
        </div>
        <AutoSizer>
          {({ height, width }) => (
            <List
              width={width}
              height={height}
              itemData={itemData}
              itemCount={visibleQueueIds.length}
              itemSize={120}
            >
              {Row}
            </List>
          )}
        </AutoSizer>
      </div>
    );
  }
}

export default QueueOptionsList;
