import React, { PureComponent, ReactNode } from 'react';
import { Form } from 'react-final-form';
import differenceWith from 'lodash/differenceWith';
import isEqual from 'lodash/isEqual';
import { YaCallsUnitKind } from 'types/YaCallsUnitKind';
import Button from '@crm/components/dist/lego2/Button';
import Textinput from '@crm/components/dist/lego2/Textinput';
import { calculatePristineByValues } from '../../../utils';
import QueueOptionsList from '../QueueOptionsList';
import Header from './Header';
import Toolbar from './Toolbar';
import { LayoutState } from './Layout.types';
import { FormValues, QueueMap } from '../types';
import { createQueueOptionKey, createQueueOptionsList } from '../utils';
import { getQueueOptions } from '../requests/getQueueOptions';
import { getQueueUnits } from '../requests/getQueueUnits';
import { setQueueOptions } from '../requests/setQueueOptions';
import css from './Layout.module.css';

class Layout extends PureComponent<{}, LayoutState> {
  public constructor(props: {}) {
    super(props);
    this.state = {
      searchText: '',
      units: [],
      queues: {
        ids: [],
        map: {},
      },
      values: {
        queueMap: {},
      },
    };
  }

  public async componentDidMount(): Promise<void> {
    const [{ queueOptions }, { units }] = await Promise.all([
      getQueueOptions(),
      getQueueUnits({ kind: YaCallsUnitKind.NonPersonal }),
    ]);

    let queueIds: number[] = [];
    let queueMap: QueueMap = {};
    queueOptions.forEach((queueOption) => {
      queueIds.push(queueOption.queueId);
      queueMap[createQueueOptionKey(queueOption.queueId)] = queueOption;
    });
    this.setState({
      queues: {
        ids: queueIds,
        map: queueMap,
      },
      values: {
        queueMap,
      },
      units,
    });
  }

  private onSubmit = (values: FormValues): Promise<unknown> | void => {
    const prevQueueOptions = createQueueOptionsList(
      this.state.queues.ids,
      this.state.values.queueMap,
    );
    const queueOptions = createQueueOptionsList(this.state.queues.ids, values.queueMap);

    const changedOptions = differenceWith(queueOptions, prevQueueOptions, isEqual);
    if (changedOptions.length) {
      const promise = setQueueOptions({ queueOptions: changedOptions });
      promise.then(() => {
        this.setState({
          values,
        });
      });

      return promise;
    }
  };

  private handleSearchChange = (text: string): void => {
    this.setState({
      searchText: text,
    });
  };

  public render(): ReactNode {
    return (
      <div className={css.Layout}>
        <div className={css.Layout__search}>
          <Textinput
            placeholder="Поиск"
            value={this.state.searchText}
            onChange={this.handleSearchChange}
          />
        </div>
        <Header />
        <div className={css.Layout__list}>
          <Form onSubmit={this.onSubmit} initialValues={this.state.values}>
            {(props) => {
              const pristine = calculatePristineByValues(
                this.state.values.queueMap,
                props.values.queueMap,
              );

              return (
                <form onSubmit={props.handleSubmit} className={css.Layout__form}>
                  <QueueOptionsList
                    units={this.state.units}
                    queueIds={this.state.queues.ids}
                    queueMap={this.state.queues.map}
                    searchText={this.state.searchText}
                  />
                  <Toolbar>
                    <Button view="action" type="submit" disabled={props.submitting || pristine}>
                      Сохранить
                    </Button>
                  </Toolbar>
                </form>
              );
            }}
          </Form>
        </div>
      </div>
    );
  }
}

export default Layout;
