import React, { PureComponent, ReactNode, ComponentClass, CSSProperties } from 'react';
import AutoSizer from 'react-virtualized-auto-sizer';
import debounce from 'lodash/debounce';
import { FixedSizeList as List } from 'react-window';
import {
  QueueOptionsListInnerProps,
  QueueOptionsListOuterProps,
  QueueOptionsListState,
} from './QueueOptionsList.types';
import QueueOption from '../QueueOption';
import { YaCallsUnit, QueueMap } from '../types';
import { createQueueOptionKey } from '../utils';
import css from './QueueOptionsList.module.css';

const Row = ({
  data,
  index,
  style,
}: {
  data: {
    units: YaCallsUnit[];
    queueMap: QueueMap;
    visibleQueueIds: number[];
  };
  index: number;
  style: CSSProperties;
}) => {
  const { units, visibleQueueIds } = data;
  const queueId = visibleQueueIds[index];
  const name = `queueMap[${createQueueOptionKey(queueId)}]`;
  return <QueueOption key={name} name={name} units={units} style={style} />;
};

class QueueOptionsList extends PureComponent<QueueOptionsListInnerProps, QueueOptionsListState> {
  public constructor(props: QueueOptionsListInnerProps) {
    super(props);
    this.state = {
      visibleQueueIds: [],
    };
  }

  private setVisibleQueueIds = debounce((searchText: string): void => {
    const visibleQueueIds: number[] = [];
    const ids = this.props.queueIds;
    const map = this.props.queueMap;
    ids.forEach((queueId) => {
      if (
        map[createQueueOptionKey(queueId)].queueName
          .toLowerCase()
          .indexOf(searchText.toLowerCase()) !== -1
      ) {
        visibleQueueIds.push(queueId);
      }
    });
    this.setState({
      visibleQueueIds,
    });
  }, 200);

  public componentDidUpdate(prevProps: QueueOptionsListInnerProps) {
    if (prevProps.searchText !== this.props.searchText) {
      this.setVisibleQueueIds(this.props.searchText);
    }

    if (prevProps.queueIds !== this.props.queueIds) {
      this.setVisibleQueueIds(this.props.searchText);
    }
  }

  public render(): ReactNode {
    const { units, queueMap } = this.props;
    const { visibleQueueIds } = this.state;
    const itemData = {
      units,
      visibleQueueIds,
      queueMap,
    };

    return (
      <div className={css.QueueOptionsList}>
        <AutoSizer>
          {({ height, width }) => (
            <List
              width={width}
              height={height}
              itemData={itemData}
              itemCount={visibleQueueIds.length}
              itemSize={34}
            >
              {Row}
            </List>
          )}
        </AutoSizer>
      </div>
    );
  }
}

export default QueueOptionsList as ComponentClass<QueueOptionsListOuterProps>;
