import { makeAutoObservable } from 'mobx';
import { get, jsonApiCall } from 'api/common';
import { YaCallsUnitKind } from 'types/YaCallsUnitKind';
import { mapErrors, mapFormValuesToApi, getFieldNameFromId } from './UnitsService.utils';
import {
  Unit,
  UnitsFormValues,
  BatchToggleIsPersonalResponse,
  BatchToggleIsPersonalRequest,
} from './UnitsService.types';

export class UnitsService {
  isLoading = false;
  units: Unit[] = [];
  initialValues: UnitsFormValues = {};

  constructor() {
    makeAutoObservable(this);
    this.load();
  }

  private load = () => {
    this.isLoading = true;

    get({
      url: `/admin/yacalls/units`,
    })
      .then((res) => {
        this.setUnits(res.units);
      })
      .finally(() => {
        this.isLoading = false;
      });
  };

  save = (units: UnitsFormValues) => {
    this.isLoading = true;

    return jsonApiCall<BatchToggleIsPersonalRequest, BatchToggleIsPersonalResponse>({
      url: '/admin/yacalls/toggleIsPersonal/batch',
      data: {
        switches: mapFormValuesToApi(units),
      },
    })
      .then((err) => {
        if (err.toggleResults?.length) return mapErrors(err.toggleResults);

        this.initialValues = units;
      })
      .finally(() => {
        this.isLoading = false;
      });
  };

  private setUnits = (units: Unit[]) => {
    this.units = units;

    const values: UnitsFormValues = {};
    for (const unit of this.units) {
      const id = getFieldNameFromId(unit.id);
      const value = String(YaCallsUnitKind[unit.kind]);
      values[id] = (value as unknown) as YaCallsUnitKind;
    }

    this.initialValues = values;
  };
}
