import * as React from 'react';
import { connect } from 'react-redux';
import memoize from 'memoize-one';
import set from 'lodash/set';
import { AnyObject } from 'react-final-form';
import redux from '../../redux';
import Form from './Form';
import getConfig from './getConfig';
import { Data, Unit } from './types';

const POSTCALL_TIMEOUT_DEFAULT = 10;
const ESCALATION_DIAL_TIMEOUT_DEFAULT = 15;

interface Props {
  id: number;
  className?: string;
  get: (id: number) => void;
  getUnits: () => void;
  post: (values: object) => object;
  data: AnyObject;
  units: Unit[];
  isLoad: boolean;
  isFetch: boolean;
  hash: number;
}

const getIntialValues = memoize((data) => ({
  ...data.user,
  channels: data.channels,
  buttonAccesses: data.buttonAccesses,
  totalLimit: data.totalLimit,
  realtimeTotalLimit: data.realtimeTotalLimit,
  ticketAssignmentMode: String(data.ticketAssignmentMode),
  ticketAssignmentModes: data.ticketAssignmentModes,
  showOwnNode: data.user.showOwnNode,
  personalUnit: data.personalUnit,
  personalPostCallTimeout: data.personalPostCallTimeout || POSTCALL_TIMEOUT_DEFAULT,
  escalationDialTimeout: data.escalationDialTimeout || ESCALATION_DIAL_TIMEOUT_DEFAULT,
  escalationUnit1: data.escalationUnitIds?.[0],
  escalationUnit2: data.escalationUnitIds?.[1],
  spQueue: data.spQueue.reduce((ac, item) => {
    // eslint-disable-next-line no-param-reassign
    ac[item.id] = item;
    return ac;
  }, {}),
}));

class FormContainer extends React.Component<Props> {
  private formRef = React.createRef<HTMLDivElement>();

  public componentDidMount() {
    this.props.get(this.props.id);
    this.props.getUnits();
  }

  public componentDidUpdate(prevProps) {
    const currentId = this.props.id;
    const lastId = prevProps.id;

    if (currentId !== lastId) {
      this.props.get(currentId);
    }
  }

  private getCheckboxValue = (item) => {
    if (item.name.startsWith('buttonAccesses')) {
      return { [item.dataset.fieldid]: item.checked };
    }
    return item.checked;
  };

  private handleSubmit = () => {
    const formNode = this.formRef.current;
    if (formNode) {
      const fieldNodes = formNode.querySelectorAll('input,select');

      const values: AnyObject = {};

      Array.prototype.forEach.call(fieldNodes, (item) => {
        let value: unknown = null;
        if (item.type === 'radio') {
          value = !values[item.name] && item.checked ? item.value : values[item.name];
        } else if (item.type === 'checkbox') {
          value = this.getCheckboxValue(item);
        } else if (item.type === 'number') {
          value = Number(item.value);
        } else {
          ({ value } = item);
        }
        set(values, item.name, value);
      });

      values.userId = this.props.data.user.userId;
      if (values.spQueue) {
        values.spQueue = Object.keys(values.spQueue).map((queueId) => ({
          id: Number(queueId),
          ...values.spQueue[queueId],
        }));
      }

      if (values.buttonAccesses) {
        values.buttonAccesses = values.buttonAccesses.reduce((acc, button) => {
          acc = { ...acc, ...button.value };
          return acc;
        }, {});
      }

      values.personalUnit = Number(values.personalUnit);
      const escalationUnit1 = Number(values.escalationUnit1);
      const escalationUnit2 = Number(values.escalationUnit2);
      values.escalationUnit1 = undefined;
      values.escalationUnit2 = undefined;
      values.escalationUnitIds = [];

      if (!values.personalUnit) {
        values.personalUnit = undefined;
        values.personalPostCallTimeout = undefined;
      }

      if (!escalationUnit1) {
        values.escalationDialTimeout = undefined;
      } else {
        values.escalationUnitIds.push(escalationUnit1);
      }
      if (escalationUnit2) {
        values.escalationUnitIds.push(escalationUnit2);
      }

      this.props.post(values);
    }
  };

  private config;

  private initialValues;

  public render(): React.ReactNode {
    const { data, units, className, hash } = this.props;

    if (!data || !units) {
      return null;
    }

    this.initialValues = getIntialValues(data);
    this.config = getConfig(data as Data, units, this.initialValues);

    return (
      <Form
        data={data}
        units={units}
        className={className}
        config={this.config}
        initialValues={this.initialValues}
        onSubmit={this.handleSubmit}
        getInnerRef={this.formRef}
        hash={hash}
      />
    );
  }
}

export default connect(
  (state) => {
    const values: AnyObject = redux.selectors.form(state) || {};
    return {
      data: values.data,
      units: values.units,
      isFetch: values.isFetch,
      isLoad: values.isLoad,
      hash: redux.selectors.hash(state),
    };
  },
  {
    get: redux.actions.form.get,
    getUnits: redux.actions.form.getUnits,
    post: redux.actions.form.post,
  },
)(FormContainer);
