import { YaCallsUnitKind } from 'types/YaCallsUnitKind';
import Access from 'utils/Access';
import { config } from 'services/Config';
import { Data, Unit } from './types';

const createPriorityItems = (channelsNumber) => {
  return [
    {
      value: 0,
      text: 'Выкл',
    },
    ...new Array(channelsNumber).fill(0).map((_, index) => ({
      value: index + 1,
      text: index + 1,
    })),
  ];
};

const filterPersonalUnits = (units: Unit[]) => {
  return units.filter((unit) => YaCallsUnitKind[unit.kind] === YaCallsUnitKind.Personal);
};

const createUnitItems = (units: Unit[]) => {
  return [{ value: 0, text: '-' }].concat(
    units.map((unit) => ({ value: unit.id, text: unit.name })),
  );
};

export const blockChildrenMapper = {
  buttonAccesses: (accesses: Data['buttonAccesses']) => {
    return accesses.map((button, index) => ({
      type: 'checkbox',
      label: button.label,
      fieldId: button.id,
      value: button.value,
      name: `buttonAccesses[${index}].value`,
    }));
  },
  personalUnitBlock: ({ units, values }) => {
    const { personalUnit, personalPostCallTimeout, escalationDialTimeout } = values;
    const escalationUnit1 = Number(values.escalationUnit1) || 0;
    const escalationUnit2 = Number(values.escalationUnit2) || 0;

    const escUnits1 = units.filter((unit) => !escalationUnit1 || unit.id !== escalationUnit2);
    const escUnits2 = units.filter((unit) => unit.id !== escalationUnit1);

    return [
      {
        type: 'select',
        label: 'Юнит для персональных звонков',
        name: 'personalUnit',
        value: Number(personalUnit),
        items: createUnitItems(filterPersonalUnits(units)),
      },
      {
        type: 'number',
        label: 'Продолжительность посткола',
        name: 'personalPostCallTimeout',
        value: personalPostCallTimeout,
      },
      {
        type: 'select',
        label: 'Юнит для эскалации 1 уровня',
        name: 'escalationUnit1',
        value: escalationUnit1,
        items: createUnitItems(escUnits1),
      },
      {
        type: 'select',
        label: 'Юнит для эскалации 2 уровня',
        name: 'escalationUnit2',
        value: escalationUnit1 ? escalationUnit2 : 0,
        items: createUnitItems(escUnits2),
      },
      {
        type: 'number',
        label: 'Продолжительность дозвона перед эскалацией',
        name: 'escalationDialTimeout',
        value: escalationDialTimeout,
      },
    ];
  },
};

export const realtimeFields = (data: Data) => [
  {
    type: 'block',
    label: 'Общие лимиты',
    fieldId: 'commonLimits',
    children: [
      {
        label: 'На все каналы',
        type: 'number',
        name: 'totalLimit',
      },
      {
        label: 'На realtime каналы',
        type: 'number',
        name: 'realtimeTotalLimit',
      },
    ],
  },
  {
    type: 'block',
    label: 'Лимиты realtime каналов',
    fieldId: 'realtimeLimits',
    children: data.channels
      .filter((channel) => channel.realtime)
      .map((channel) => ({
        type: 'number',
        label: channel.name,
        name: `channels[${data.channels.findIndex((item) => item.type === channel.type)}].limit`,
      })),
  },
  {
    type: 'block',
    label: 'Лимиты не-realtime каналов',
    fieldId: 'nonRealtimeLimits',
    children: data.channels
      .filter((channel) => !channel.realtime)
      .map((channel) => ({
        type: 'number',
        label: channel.name,
        name: `channels[${data.channels.findIndex((item) => item.type === channel.type)}].limit`,
      })),
  },
];

export const getConfig = (data: Data, units: Unit[], initialValues: unknown) => ({
  fields: [
    {
      label: 'Старший менеджер',
      type: 'checkbox',
      name: 'groupLead',
    },
    {
      label: 'Главная роль',
      type: 'select',
      name: 'mainRoleId',
      disabled: !Access.isEdit(data.props.mainRole),
      items: data.sysRoles.map((role) => ({ value: role.id, text: role.name || '' })),
    },
    {
      type: 'checkbox-group',
      label: 'Роли',
      disabled: !Access.isEdit(data.props.roles),
      items: data.sysRoles.map((role) => ({ name: role.code, text: role.name || '' })),
    },
    {
      label: 'МКС офис',
      type: 'select',
      name: 'spCityId',
      items: [{ value: '' as number | string, text: '' }].concat(
        data.sysCities.map((city) => ({ value: city.id, text: city.name || '' })),
      ),
    },
    {
      label: 'SpRoleType',
      type: 'text',
      name: 'spRoleType',
    },
    {
      label: 'spMaxTicketAllowed',
      type: 'text',
      name: 'spMaxTicketAllowed',
    },
    {
      label: 'spOfficeSalePhoneNumber',
      type: 'text',
      name: 'spOfficeSalePhoneNumber',
    },
    {
      label: 'Алиасы',
      type: 'text',
      name: 'crmEmails',
    },
    {
      label: 'Логин во внешнем Паспорте',
      type: 'text',
      name: 'crmYndxLogin',
    },
    {
      label: 'Не может быть менеджером клиента',
      type: 'checkbox',
      name: 'cantBeManager',
    },
    {
      label: 'Есть папка Мои',
      type: 'checkbox',
      name: 'showOwnNode',
    },
    {
      label: 'Режим получения тикетов и сделок',
      type: 'radio-group',
      name: 'ticketAssignmentMode',
      items: data.ticketAssignmentModes,
    },
    {
      type: 'block',
      label: 'Подключенные каналы',
      fieldId: 'connectedChannels_checkbox',
      children: data.channels.map((channel, index) => ({
        type: 'checkbox',
        label: channel.name,
        name: `channels[${index}].on_off_manual`,
      })),
    },
    {
      type: 'block',
      label: 'Подключенные каналы',
      fieldId: 'connectedChannels_select',
      children: data.channels.map((channel, index, array) => ({
        type: 'select',
        label: channel.name,
        name: `channels[${index}].priority`,
        items: createPriorityItems(array.length),
      })),
    },
    {
      type: 'block',
      label: 'Лимиты на каналы',
      fieldId: 'channelsLimits',
      children: [
        {
          label: 'Общий лимит',
          type: 'number',
          name: 'totalLimit',
        },
      ].concat(
        data.channels.map((channel, index, array) => ({
          type: 'number',
          label: channel.name,
          name: `channels[${index}].limit`,
          items: createPriorityItems(array.length),
        })),
      ),
    },
    {
      type: 'block',
      label: 'Персональный юнит',
      description: `Наличие персонального юнита позволяет:
      <ol><li>Совершать исходящие звонки из карточки аккаунта, сделок без привязки к очереди, сделок в очередях, не привязанных к телефонии, планируемых звонков. При этом персональный юнит не используется для звонков из тикетного модуля.</li>
      <li>Использовать поп-ап звонка во время разговора для добавления комментария и категорий, привязки сделок к звонку.</li></ol>`,
      fieldId: 'personalUnitBlock',
      name: 'personalUnitBlock',
      children: blockChildrenMapper.personalUnitBlock({ units, values: initialValues }),
    },
    {
      type: 'block',
      label: 'Доступ к кнопкам',
      fieldId: 'buttonAccesses',
      name: 'buttonAccesses',
      children: blockChildrenMapper.buttonAccesses(data.buttonAccesses),
    },
    {
      label: 'МКС Очереди',
      type: 'queue',
      items: data.spQueue,
      name: 'spQueue',
    },
  ].concat(
    data.channels.map((channel, index) => ({
      type: 'number',
      hidden: true,
      label: channel.name,
      name: `channels[${index}].type`,
    })),
  ),
});

export default (data: Data, units: Unit[], initialValues: unknown) => {
  const fieldsConfig = getConfig(data, units, initialValues);
  if (config.value.features.newRealtimeLimit) {
    const channelsLimitsFieldIndex = fieldsConfig.fields.findIndex(
      ({ fieldId }) => fieldId === 'channelsLimits',
    );
    fieldsConfig.fields.splice(channelsLimitsFieldIndex, 1, ...realtimeFields(data));
  }

  return fieldsConfig;
};
