import cx from 'classnames';
import { AnyObject } from 'final-form';
import get from 'lodash/get';
import cssBase from './Form.scss';
import { renderBlock } from './renderBlock';
import { idPrefix } from './utils/idPrefix';
export interface Template {
  componentDidMount: (() => void)[];
  html: string;
}

let template: Template;

interface RowProps {
  css: {
    [key: string]: string;
  };
  label: string;
  children?: string | unknown[];
  column?: boolean;
  name?: string;
}

const renderRow = (props: RowProps): string => {
  const { css, label, children, column, name = '' } = props;

  return `<div id="${idPrefix}_${name}" class="${cx(css.b__row, {
    [css.b__row_column]: column,
  })}"><div class="${css.b__label}">${label}</div><div class="${
    css.b__field
  }">${children}</div></div>`;
};

export interface InputProps extends RowProps {
  type: string;
  name: string;
  value: AnyObject;
  hidden?: boolean;
  fieldId?: string;
  useOriginalValue?: boolean;
}

const queueToggleLogic = (name: string, slug: string) => {
  const button = document.querySelector(`button[data-id="${name}.${slug}"]`);
  if (button) {
    button.addEventListener('click', () => {
      const inputs = document.querySelectorAll<HTMLInputElement>(
        `input[data-id="${name}.${slug}"]`,
      );
      if (inputs) {
        const checked = !inputs[0].checked;
        Array.prototype.forEach.call(inputs, (input): void => {
          // eslint-disable-next-line no-param-reassign
          input.checked = checked;
        });
      }
    });
  }
};

const renderQueue = (props): string => {
  const { items, css, name, value = {}, ...other } = props;

  template.componentDidMount.push((): void => {
    queueToggleLogic(name, 'canTake');
    queueToggleLogic(name, 'canList');
    queueToggleLogic(name, 'isQueueMember');
  });
  return renderRow({
    ...other,
    css,
    children: `<div><div>
        <button data-id="${name}.canTake">Выделить/Снять CanTake</button>
        <button data-id="${name}.canList">Выделить/Снять CanList</button>
        <button data-id="${name}.isQueueMember">Выделить/Снять isQueueMember</button>
        </div>${items
          .map(
            (q) =>
              `<div class="${css.b__queue}"><span>${q.code}</span><span>${
                q.name
              }</span><span><label><input data-id="${name}.isQueueMember" name="${name}.${
                q.id
              }.isQueueMember" type="checkbox" ${
                (value[q.id] || {}).isQueueMember ? 'checked="checked"' : ''
              }/>isQueueMember</label></span><span><label><input data-id="${name}.canTake" name="${name}.${
                q.id
              }.canTake" type="checkbox" ${
                (value[q.id] || {}).canTake ? 'checked="checked"' : ''
              }/>CanTake</label></span><span><label><input data-id="${name}.canList" name="${name}.${
                q.id
              }.canList" type="checkbox" ${
                (value[q.id] || {}).canList ? 'checked="checked"' : ''
              }/>CanList</label></span><span>takePriority:<input type="number" class="${
                css.b__takePriorityInput
              }" name="${name}.${q.id}.takePriority" type="number" value="${(value[q.id] || {})
                .takePriority || ''}"/></span></div>`,
          )
          .join('')}</div>`,
  });
};

const renderCheckBoxGroup = (props): string => {
  const { items, value = {}, disabled, ...other } = props;

  return renderRow({
    ...other,
    children: items
      .map(
        (item) =>
          `<div><label><input name="${item.name}" type="checkbox" ${
            value[item.name] ? 'checked="checked"' : ''
          }" ${disabled ? 'disabled' : ''}/>${item.text}</label></div>`,
      )
      .join(''),
  });
};

const renderRadioGroup = (props): string => {
  const { items, value = {}, type, name, ...other } = props;
  const inputValue = get(value, name);
  return renderRow({
    ...other,
    children: items
      .map(
        (item) =>
          `<div>
            <label>
            <input name="${name}" value="${item.id}" type="radio" ${
            inputValue === item.id ? 'checked' : ''
          }/>
            ${item.name}
            </label>
            </div>`,
      )
      .join(''),
  });
};

const renderSelect = (props): string => {
  const { items, name, value, disabled, ...other } = props;

  return renderRow({
    ...other,
    name,
    children: `<select name="${name}" ${disabled ? 'disabled' : ''}>${items
      .map(
        (item) =>
          `<option value="${item.value}" ${
            value === item.value && item.text ? 'selected="selected"' : ''
          }>${item.text}</option>`,
      )
      .join('')}</select>`,
  });
};

export function renderInput(props: InputProps): string {
  const { type, name, value = {}, hidden, useOriginalValue = false, ...other } = props;
  const inputValue = useOriginalValue ? value : get(value, name);
  if (hidden) {
    return `<div hidden>${renderInput({ ...props, hidden: false })}</div>`;
  }

  if (type === 'select') {
    return renderSelect({ ...props, value: inputValue });
  }

  if (type === 'checkbox-group') {
    return renderCheckBoxGroup(props);
  }

  if (type === 'radio-group') {
    return renderRadioGroup(props);
  }

  if (type === 'queue') {
    return renderQueue({ ...props, value: inputValue });
  }

  if (type === 'checkbox') {
    return renderRow({
      ...other,
      children: `<input ${
        props.fieldId ? `data-fieldId="${props.fieldId}"` : ''
      } name="${name}" type="${type}" ${inputValue ? 'checked="checked"' : ''} />`,
    });
  }

  if (type === 'block') {
    return renderBlock({
      ...other,
      value,
      type,
      label: props.label,
      children: props.children as AnyObject[],
    });
  }

  return renderRow({
    ...other,
    name,
    children: `<input name="${name}" type="${type}" value="${
      inputValue != null ? inputValue : ''
    }" />`,
  });
}

export default (props): Template => {
  template = {
    componentDidMount: [],
    html: '',
  };

  template.html = `<div>${props.config.fields
    .map((field): string =>
      renderInput({
        ...field,
        css: cssBase,
        value: props.initialValues,
      }),
    )
    .join('')}</div>`;

  return template;
};
