import { RefObject } from 'react';
import css from '../NativeForm.module.css';
import { idPrefix } from './idPrefix';
import { renderBlock, RenderBlockProps } from '../renderBlock';
import { blockChildrenMapper } from '../getConfig';
import cssBase from '../Form.scss';
import { updateFetcher } from './updateFetcher';
import { getHiddenFields } from './getHiddenFields';
import { Data, Unit } from '../types';

const hiddenFields = new Set(['connectedChannels_checkbox']);

const fieldsForHideMap = {
  ticketAssignmentMode: {
    '1': [
      'connectedChannels_select',
      'channelsLimits',
      'commonLimits',
      'realtimeLimits',
      'nonRealtimeLimits',
    ],
    '2': ['spMaxTicketAllowed', 'connectedChannels_checkbox'],
    '3': [
      'connectedChannels_select',
      'connectedChannels_checkbox',
      'channelsLimits',
      'commonLimits',
      'realtimeLimits',
      'nonRealtimeLimits',
      'spMaxTicketAllowed',
      'buttonAccesses',
    ],
  },
};

const fieldsForShowMap = {
  ticketAssignmentMode: {
    '1': ['connectedChannels_checkbox'],
  },
};

const needListen = [
  'ticketAssignmentMode',
  'channels[1].on_off_manual',
  'channels[2].on_off_manual',
  'channels[1].priority',
  'channels[2].priority',
  'personalUnit',
  'personalPostCallTimeout',
  'escalationDialTimeout',
  'escalationUnit1',
  'escalationUnit2',
];

const needUpdate = {
  ticketAssignmentMode: ['buttonAccesses'],
  escalationUnit1: ['personalUnitBlock'],
  escalationUnit2: ['personalUnitBlock'],
};

export class FormFieldsController {
  private formRef: RefObject<HTMLDivElement> | undefined;
  private config: unknown | undefined;
  private initialValues: unknown | undefined;
  private data: Data | undefined;
  private units: Unit[] | undefined;
  private hiddenNodesMap = new Map();
  private formValues = {};
  constructor(formRef: RefObject<HTMLDivElement>, { config, initialValues, data, units }) {
    this.formRef = formRef;
    this.config = config;
    this.initialValues = initialValues;
    this.data = data;
    this.units = units;
  }
  private getNodes = (FieldsIds: string[]): NodeListOf<HTMLDivElement> | undefined => {
    const query = FieldsIds.map((id) => `#${idPrefix}_${id}`).join(',');
    return this.formRef?.current?.querySelectorAll(query);
  };

  private getNodeByFieldId = (fieldId: string) => {
    const id = `${idPrefix}_${fieldId}`;
    return this.formRef?.current?.querySelector(`#${id}`);
  };

  private hideNodes = (nodes: NodeListOf<HTMLDivElement> | undefined): void => {
    if (nodes) {
      nodes.forEach((node) => {
        node.classList.add(css.NativeForm__hiddenRow);
      });
    }
  };

  private updateNode = async (evt, fieldName: string) => {
    const updates = await updateFetcher[fieldName]?.({
      evt,
      initialValues: this.initialValues as { userId: number },
      values: this.formValues,
      units: this.units,
    });

    const targetNode = this.getNodeByFieldId(fieldName);
    if (!targetNode) return;

    const children = blockChildrenMapper[fieldName](updates);
    const newNode = this.getNodeHtml(fieldName, children);
    this.replaceNode(targetNode, newNode);

    this.setInitialFormValues();
  };

  private getNodeHtml = (fieldName: string, children) => {
    if (!children?.length) return `<div id="${idPrefix}_${fieldName}"></div>`;

    const config = this.config as { fields: { name: string }[] } | undefined;
    const props = config?.fields.find((field) => field.name === fieldName);

    return renderBlock({
      ...((props as unknown) as RenderBlockProps),
      css: cssBase,
      children,
      useOriginalValue: true,
    });
  };

  private replaceNode = (targetNode: Element, newNode): void => {
    const tempContainer = document.createElement('div');
    tempContainer.innerHTML = newNode;
    const node = tempContainer.firstElementChild;
    targetNode.replaceWith(node as Node);
  };

  private showNodes = (nodes: NodeListOf<HTMLDivElement> | undefined): void => {
    if (nodes) {
      nodes.forEach((node) => {
        node.classList.remove(css.NativeForm__hiddenRow);
      });
    }
  };

  /**
   * Вся эта красота сейчас работает только с 1 инициатором скрытия полей,
   * если добавлять еще N инициаторов надо придумывать методы разруливания логики
   * сокрытия и показа полей.
   * То есть если несколько полей с текущим занчением хотят скрыть какое то поле,
   * а другие наоборот хотят его показать. Верю что рефаторинг формы слуится раньше.
   */
  private nodeListener = async (initiator, value, evt) => {
    const hiddenGroups = getHiddenFields({ initiator, form: this.formValues });

    if (needUpdate[initiator]) {
      await Promise.all(needUpdate[initiator].map((fieldName) => this.updateNode(evt, fieldName)));
    }

    for (const group of hiddenGroups) {
      this.updateGroupVisibility(group.groupName, group.hiddenIds);
    }

    const hiddenIds = fieldsForHideMap[initiator] && fieldsForHideMap[initiator][value];
    this.updateGroupVisibility(initiator, hiddenIds);
  };

  private updateGroupVisibility = (groupName, hiddenIds) => {
    if (this.hiddenNodesMap.has(groupName)) {
      this.showNodes(this.hiddenNodesMap.get(groupName));
    }

    if (hiddenIds?.length) {
      const nodesForHidden = this.getNodes(hiddenIds);
      if (nodesForHidden) {
        this.hideNodes(nodesForHidden);
        this.hiddenNodesMap.set(groupName, nodesForHidden);
      }
    }
  };

  public setInitialFormValues = () => {
    const selector = needListen
      .map((name) => `input[name="${name}"],select[name="${name}"]`)
      .join(',');

    const fields = Array.from(
      this.formRef?.current?.querySelectorAll(selector) || [],
    ) as HTMLInputElement[];

    const values = this.formValues;
    for (const item of fields) {
      if (item.type === 'radio') {
        values[item.name] = !values[item.name] && item.checked ? item.value : values[item.name];
      } else if (item.type === 'checkbox') {
        values[item.name] = item.checked;
      } else if (item.type === 'number') {
        values[item.name] = Number(item.value);
      } else {
        values[item.name] = item.value;
      }
    }
  };

  public setNodeListeners = () => {
    this.formRef?.current?.addEventListener('change', (evt) => {
      const { name, type, value, checked } = evt.target as HTMLInputElement;

      if (needListen.includes(name)) {
        const val = type === 'checkbox' ? checked : value;
        this.formValues[name] = val;

        this.nodeListener(name, val, evt);
      }
    });
  };

  public initialSetFieldsVisibility = (fieldsValues) => {
    this.setInitialFormValues();

    const hiddenGroups = getHiddenFields({ form: this.formValues, checkAll: true });
    for (const group of hiddenGroups) {
      this.updateGroupVisibility(group.groupName, group.hiddenIds);
    }

    for (let [key, value] of Object.entries(fieldsValues)) {
      if (fieldsForHideMap[key] && fieldsForHideMap[key][value]) {
        this.updateGroupVisibility(key, fieldsForHideMap[key][value]);
        if (fieldsForShowMap[key] && fieldsForShowMap[key][value]) {
          fieldsForShowMap[key][value].forEach((id) => hiddenFields.delete(id));
        }
      }
    }
    if (hiddenFields.size !== 0) {
      this.hideNodes(this.getNodes([...hiddenFields]));
    }
  };
}
