import * as React from 'react';
import Promise from 'bluebird';
import { Form as RFF } from 'react-final-form';
import Button from '@crm/components/dist/lego2/Button';
import cx from 'classnames';
import { Field as IField, FieldComponent } from '../types';
import FormTextInput from './FormTextInput';
import FormCheckBox from './FormCheckBox';
import css from './Form.module.css';

interface Props<T> {
  onSubmit: (values, formikBag) => Promise<T>;
  className?: string;
  fields?: IField[];
}

const components = {
  [FieldComponent.TextInput]: FormTextInput,
  [FieldComponent.CheckBox]: FormCheckBox,
};

interface FieldsContent {
  [component: string]: React.ReactElement[];
}

type GetFieldsContent = (fields: IField[] | undefined) => FieldsContent;
const getFieldsContent: GetFieldsContent = (fields) => {
  if (!Array.isArray(fields) || !fields.length) {
    return {};
  }

  return fields.reduce((acc, field, index) => {
    const Component = components[field.component];
    if (!Array.isArray(acc[field.component])) {
      acc[field.component] = [<Component key={index} field={field} />];
      return acc;
    }

    acc[field.component].push(<Component key={index} field={field} />);
    return acc;
  }, {});
};

type Form = <T extends {}>(props: React.PropsWithChildren<Props<T>>) => React.ReactElement;
const Form: Form = (props) => {
  const { onSubmit, className, fields } = props;

  const fieldsContent: FieldsContent = getFieldsContent(fields);
  const hasCheckBoxes =
    Array.isArray(fieldsContent[FieldComponent.CheckBox]) &&
    fieldsContent[FieldComponent.CheckBox].length;

  return (
    <RFF
      onSubmit={onSubmit}
      initialValues={{}}
      render={(formikProps) => (
        <form onSubmit={formikProps.handleSubmit} className={cx(className, css.b)}>
          <div className={css.b__row}>
            {fieldsContent[FieldComponent.TextInput]}
            <Button type="submit" view="action" disabled={formikProps.submitting}>
              Найти
            </Button>
          </div>
          {hasCheckBoxes && (
            <div className={css.b__row}>{fieldsContent[FieldComponent.CheckBox]}</div>
          )}
        </form>
      )}
    />
  );
};

export default Form;
