import * as React from 'react';
import Promise from 'bluebird';
import { get } from 'api/common';
import cx from 'classnames';
import Loadable from 'utils/Loadable';
import applyMixins from 'utils/applyMixins';
import Spinner from 'components/Spinner';
import Form from './Form';
import Table from './Table';
import css from './Report.module.css';
import { Config, Response, Table as ITable } from '../types';

interface Props {
  config: Config;
  className?: string;
}

interface State {
  items?: ITable[];
  scrollTop: number;
  lastValues?: object;
  isFetch: boolean;
  isLoad: boolean;
}

class Report extends React.Component<Props, State> {
  private promise: Promise<Response>;

  public state: State = { scrollTop: 0, isFetch: false, isLoad: false };

  public componentWillUnmount(): void {
    // @ts-ignore
    this.loadCancel();
  }

  private handleSubmit = (values): Promise<Response> => {
    const { config } = this.props;

    // @ts-ignore
    this.loadCancel();
    this.setState({ items: undefined, isFetch: true, lastValues: values, isLoad: false });
    this.promise = get({
      url: config.requestUrl,
      data: values,
      global: false,
    })
      .finally(() => {
        this.setState({ isFetch: false, isLoad: true });
      })
      .then((data) => {
        this.setState({ items: data.tables });
        return data;
      }) as Promise<Response>;

    return this.promise;
  };

  private handleLoadMore = (params) => {
    const { lastValues } = this.state;

    return get({
      url: '/view/billing/table',
      data: {
        ...lastValues,
        ...params,
      },
      global: false,
    });
  };

  private handleScroll = (e): void => {
    if (e.target === e.currentTarget) {
      this.setState({ scrollTop: e.target.scrollTop });
    }
  };

  public render(): React.ReactElement | null {
    const { config, className } = this.props;

    if (!config) {
      return null;
    }

    const { title, fields } = config;
    const { items, scrollTop, isFetch, isLoad } = this.state;

    let content;
    if (isFetch) {
      content = <Spinner visible modal={false} />;
    } else if (Array.isArray(items)) {
      content = items.map((item, index) => (
        <div key={index}>
          <h2 className={css.b__h2}>{item.title}</h2>
          <Table className={css.b__table} table={item} onLoad={this.handleLoadMore} />
        </div>
      ));
    } else if (isLoad) {
      content = 'Нет данных';
    }

    return (
      <div className={cx(css.b, className)} onScroll={this.handleScroll}>
        <div className={cx(css.b__header, { [css.b__header_above]: scrollTop })}>
          <h1 className={css.b__h1}>{title}</h1>
          <Form fields={fields} onSubmit={this.handleSubmit} />
        </div>
        <div className={css.b__tables}>{content}</div>
      </div>
    );
  }
}

applyMixins(Report, [Loadable]);

export default Report;
