import * as React from 'react';
import Promise from 'bluebird';
import Button from '@crm/components/dist/lego2/Button';
import Loadable from 'utils/Loadable';
import applyMixins from 'utils/applyMixins';
import { Table as ITable, ResponseMore } from '../types';

interface Props {
  onLoad: (param: { tableId: string; offset: number }) => any;
  className?: string;
  table: ITable;
}

interface State {
  html: string;
  eof: boolean;
  isFetch: boolean;
}

class Table extends React.Component<Props, State> {
  public constructor(props) {
    super(props);

    const table = this.props.table || {};

    this.state = { html: String(table.html || '').trim(), eof: table.eof, isFetch: false };
  }

  public componentWillUnmount(): void {
    this.loadCancel();
  }

  private getOffset(): number {
    const div = document.createElement('div');
    div.innerHTML = this.state.html;

    const table = div.firstChild as HTMLTableElement;
    return table.querySelectorAll('tr').length - 1;
  }

  private handleLoadMore = (): Promise<ResponseMore> => {
    const { onLoad } = this.props;

    this.loadCancel();
    this.setState({ isFetch: true });
    this.promise = onLoad({
      tableId: this.props.table.id,
      offset: this.getOffset(),
    })
      .finally(() => {
        this.setState({ isFetch: false });
      })
      .then((data) => {
        this.setState({
          eof: data.eof,
          html: this.updateHtml(data.html),
        });

        return data;
      }) as Promise<ResponseMore>;

    return this.promise;
  };

  private updateHtml(html: string): string {
    const div = document.createElement('div');

    div.innerHTML = String(html || '').trim();
    const loadedTable = div.firstChild as HTMLTableElement;

    div.innerHTML = this.state.html;
    const table = div.firstChild as HTMLTableElement;

    const tbody = loadedTable.querySelector('tbody');
    if (tbody) {
      table.append(tbody);
    }

    return div.innerHTML;
  }

  public render(): React.ReactElement {
    const { className } = this.props;
    const { html, eof, isFetch } = this.state;

    return (
      <div>
        <div dangerouslySetInnerHTML={{ __html: html }} className={className} />
        {!eof && (
          <div>
            <Button onClick={this.handleLoadMore} progress={isFetch}>
              Загрузить еще
            </Button>
          </div>
        )}
      </div>
    );
  }
}

// eslint-disable-next-line @typescript-eslint/no-empty-interface
interface Table extends Loadable {}
applyMixins(Table, [Loadable]);

export default Table;
