import React, { RefObject } from 'react';
import { Tabs, Tab } from 'lego/components/Tabs';
import ToggleButton from 'lego/components/ButtonToggle';
import cx from 'classnames';
import Metrika from 'modules/metrika';
import resize from 'utils/resize';
import { CategoriesSelectedList, CategoriesSelectedTags } from '../CategoriesSelected';
import CategoriesList from '../CategoriesList';
import { CategoriesTree } from '../CategoriesTree';
import css from './Categorization.module.css';
import { tipCallbackContext } from '../TipButton';
import {
  WithSearchController,
  CategorizationProps,
  CategorizationState,
} from './Categorization.types';

export class Categorization extends React.Component<CategorizationProps, CategorizationState> {
  static defaultProps = {
    onTipClick: () => {},
    isRequireMarkup: 0,
  };

  static readonly TABS = {
    0: 'List',
    1: 'Tree',
  };

  private readonly listRef: RefObject<WithSearchController>;
  private readonly treeRef: RefObject<WithSearchController>;
  private searchText: string;

  constructor(props) {
    super(props);
    this.state = {
      isOpen: Boolean(props.isOpen),
      tabIndex: 0,
    };

    this.listRef = React.createRef();
    this.treeRef = React.createRef();
    this.searchText = '';
  }

  handleToggleClick = () => {
    this.toggle();
  };

  close() {
    Metrika.reachGoal('categorizator_collapse');
    const { onClose } = this.props;
    if (onClose) {
      onClose();
    }

    this.setState(
      {
        isOpen: false,
      },
      resize,
    );
  }

  open() {
    Metrika.reachGoal('categorizator_expand');
    const { onOpen } = this.props;
    if (onOpen) {
      onOpen();
    }

    this.setState(
      {
        isOpen: true,
      },
      resize,
    );
  }

  toggle() {
    const { onToggle } = this.props;
    if (onToggle) {
      onToggle(!this.state.isOpen);
    }

    if (this.state.isOpen) {
      this.close();
    } else {
      this.open();
    }
  }

  handleTabsChange = (prevTabIndex, nextTabIndex) => {
    switch (prevTabIndex) {
      case 0: {
        this.searchText = this.listRef.current!.getText();
        break;
      }

      case 1: {
        this.searchText = this.treeRef.current!.getText();
        break;
      }

      default:
        break;
    }

    switch (nextTabIndex) {
      case 0: {
        this.setState({ tabIndex: 0 }, () => {
          Metrika.reachGoal('categorizator_list_open');
        });
        this.listRef.current!.setText(this.searchText);
        break;
      }

      case 1: {
        this.setState({ tabIndex: 1 }, () => {
          Metrika.reachGoal('categorizator_tree_open');
        });
        this.treeRef.current!.setText(this.searchText);
        break;
      }

      default:
        break;
    }
  };

  handleChangeTab = (prevTabIndex, nextTabIndex) => {
    resize();
    this.handleTabsChange(prevTabIndex, nextTabIndex);
  };

  render() {
    const { className, theme, onTipClick, name, isRequireMarkup } = this.props;
    const { isOpen, tabIndex } = this.state;
    const classNames = cx(className, css.layoutWrap, {
      [css.openLayoutWrap]: isOpen,
    });
    return (
      <tipCallbackContext.Provider value={onTipClick}>
        <div className={classNames}>
          <Metrika.Store categorizator_view={Categorization.TABS[tabIndex]} />
          <header className={css.layoutHeader}>
            <span className={css.layoutTitle}>Категория</span>
            <div className={css.headerSelectedCategories}>
              {!isOpen && <CategoriesSelectedTags isRequireMarkup={isRequireMarkup} />}
            </div>
            <span className={css.layoutToggleWrap}>
              <ToggleButton open={isOpen} view="clear" onClick={this.handleToggleClick}>
                <span className={css.layoutToggleText}>{isOpen ? 'Свернуть' : 'Развернуть'}</span>
              </ToggleButton>
            </span>
          </header>
          <div className={cx(theme, css.layout)}>
            <div className={css.mainSelectedCategories}>{isOpen && <CategoriesSelectedList />}</div>
            <Tabs navigationTheme="start" onDidUpdate={this.handleChangeTab}>
              <Tab title="Список" key="list" className={css.b__tabWrap} keepMount lazyMount>
                <CategoriesList className={css.b__tab} ref={this.listRef} name={name} />
              </Tab>
              <Tab title="Дерево" key="tree" className={css.b__tabWrap} keepMount lazyMount>
                <CategoriesTree
                  className={cx(css.b__tab, css.b__tab_categories)}
                  name={name}
                  getRef={this.treeRef}
                />
              </Tab>
            </Tabs>
          </div>
        </div>
      </tipCallbackContext.Provider>
    );
  }
}
