import { createSlice, createAsyncThunk, PayloadAction } from '@reduxjs/toolkit';
import lodashGet from 'lodash/get';
import { get } from 'api/common';
import { loadList } from '../api';
import {
  CategorizationValue,
  CategoryAddPayload,
  CategoryRemovePayload,
  VALUE_FROM_TYPE_KEY,
  ValueFromType,
  CATEGORAZATION_V3,
} from '../types';

const arrToObj = (arr) =>
  arr.reduce((acc, item) => {
    acc[item] = item;
    return acc;
  }, {});

export interface CategorizationSliceOptions {
  workplace: string;
  issueId?: number;
  eId?: number;
}

export function createCategorizationSlice(name: string, options: CategorizationSliceOptions) {
  const sliceName = `categorization/${name}`;

  const loadCategory = createAsyncThunk(
    `${sliceName}/loadCategory`,
    async (payload: { categoryId: number }) => {
      const { categoryId } = payload;

      const categories = await get({
        url: '/view/category/tree',
        data: {
          categoryIds: [categoryId],
          ...options,
        },
      });

      return { categoryId, category: categories.items[0] };
    },
  );

  const set = (state, { payload }: PayloadAction<CategoryAddPayload>) => {
    const { categoryId, category } = payload;
    const parentCategoryId = category && category.id;
    let updateCategories;
    if (parentCategoryId && !state.categories[parentCategoryId]) {
      updateCategories = {
        ...state.categories,
        [parentCategoryId]: category,
      };
    }

    let updateSelectedCategories;
    if (!state.selectedCategories[parentCategoryId]) {
      updateSelectedCategories = {
        ...state.selectedCategories,
        [parentCategoryId]: {},
      };
    }

    if (categoryId !== parentCategoryId) {
      if (Array.isArray(categoryId)) {
        const objCategories = arrToObj(categoryId);
        if (updateSelectedCategories) {
          updateSelectedCategories[parentCategoryId] = objCategories;
        } else {
          updateSelectedCategories = {
            ...state.selectedCategories,
            [parentCategoryId]: objCategories,
          };
        }
      } else if (updateSelectedCategories) {
        updateSelectedCategories[parentCategoryId][categoryId] = categoryId;
      } else {
        updateSelectedCategories = {
          ...state.selectedCategories,
          [parentCategoryId]: {
            ...state.selectedCategories[parentCategoryId],
            [categoryId]: categoryId,
          },
        };
      }
    }

    return {
      ...state,
      categories: updateCategories || state.categories,
      selectedCategories: updateSelectedCategories || state.selectedCategories,
      [VALUE_FROM_TYPE_KEY]: ValueFromType.Inside,
    };
  };

  const slice = createSlice({
    name: sliceName,
    initialState: {
      categories: {},
      selectedCategories: {},
      [VALUE_FROM_TYPE_KEY]: ValueFromType.Inside,
    } as CategorizationValue,
    reducers: {
      setValue(state, { payload = {} }) {
        return {
          ...state,
          ...payload,
          [VALUE_FROM_TYPE_KEY]: ValueFromType.Inside,
          [CATEGORAZATION_V3]: false,
        };
      },
      setValueFromOutside(state, { payload = {} }) {
        return {
          ...state,
          ...payload,
          [VALUE_FROM_TYPE_KEY]: ValueFromType.Outside,
          [CATEGORAZATION_V3]: false,
        };
      },
      set,
      remove(state, { payload }: PayloadAction<CategoryRemovePayload>) {
        const { categoryId, subCategoryId } = payload;

        if (subCategoryId) {
          const selectedCategories = {
            ...state.selectedCategories,
            [categoryId]: {
              ...state.selectedCategories[categoryId],
            },
          };

          delete selectedCategories[categoryId][subCategoryId];

          return {
            ...state,
            selectedCategories,
            [CATEGORAZATION_V3]: false,
          };
        }

        const categories = { ...state.categories };
        const selectedCategories = { ...state.selectedCategories };
        delete categories[categoryId];
        delete selectedCategories[categoryId];

        return {
          ...state,
          categories,
          selectedCategories,
          [VALUE_FROM_TYPE_KEY]: ValueFromType.Inside,
          [CATEGORAZATION_V3]: false,
        };
      },
      removeV3(state, { payload }) {
        const { categoryId } = payload;
        const selectedCategories = { ...state.selectedCategories };
        const categories = { ...state.categories };
        delete categories[categoryId];
        delete selectedCategories[categoryId];

        return {
          ...state,
          categories,
          selectedCategories,
          [VALUE_FROM_TYPE_KEY]: ValueFromType.Inside,
          [CATEGORAZATION_V3]: true,
        };
      },
    },
    extraReducers: {
      [loadCategory.fulfilled.toString()]: set,
    },
  });

  const getState = (state) => state[sliceName];
  const getCategoryValue = (state, id: number) =>
    lodashGet(state, [sliceName, 'selectedCategories', id]);
  const getSelectedMap = (state) => {
    const selectedCategories = lodashGet(state, [
      sliceName,
      'selectedCategories',
    ]) as CategorizationValue['selectedCategories'];
    return Object.entries(selectedCategories).reduce((acc, [key, value]) => {
      acc[key] = true;
      Object.keys(value).forEach((k) => {
        acc[k] = true;
      });

      return acc;
    }, {});
  };

  return {
    loadCategory,
    slice,
    reducer: slice.reducer,
    actions: slice.actions,
    getState,
    getCategoryValue,
    getSelectedMap,
    treeApi: `/view/category/tree?${new URLSearchParams({
      workplace: options.workplace,
      issueId: String(options.issueId || ''),
      eId: String(options.eId || ''),
    }).toString()}`,
    loadList: (args) => loadList({ ...args, ...options }),
  };
}
