import { makeObservable, observable, action, computed, reaction } from 'mobx';
import { AlertDTO } from 'types/dto/AlertDTO';
import {
  AlertStore as IAlertStore,
  AlertLoadHandler,
  Tree,
  Store,
  EmittingPayload,
} from '../../../types';
import { BoolState } from '../BoolState';

export class AlertStore implements IAlertStore {
  private externalHandler: AlertLoadHandler = () => Promise.resolve([]);

  public byCategoryId: Map<number, AlertDTO | undefined> = new Map();

  public constructor() {
    makeObservable(this, {
      load: action.bound,
      canBeVisibleAlert: computed,
      visibleAlert: computed,
      hasRelatedAlerts: computed,
      closestId: computed,
      relatedAlerts: computed,
      byCategoryId: observable,
      reset: action.bound,
    });
  }

  private tree: Tree;

  private main: Store;

  public setStores(stores: { tree: Tree; main: Store }) {
    this.tree = stores.tree;
    this.main = stores.main;
  }

  public runReactions() {
    return [
      reaction(
        () => this.tree.lastHighlighted,
        (lastHighlighted) => {
          if (lastHighlighted == null) {
            this.byCategoryId.clear();
            return;
          }

          this.load(this.tree.highlightPath);
        },
      ),
    ];
  }

  public expanding = new BoolState(true);

  public load(categoryIds: number[]) {
    this.externalHandler(categoryIds)
      .then(
        action((alerts = []) => {
          this.byCategoryId = new Map(alerts.map((alert) => [alert.category.id, alert]));
          const emittingPayload: EmittingPayload = {
            hasAlert: alerts.length > 0,
            categoryId: this.tree.lastHighlighted,
          };
          if (emittingPayload.hasAlert) {
            emittingPayload.categoryIdHasAlert = this.closestId;
          }

          this.main.emit('alertLoadSuccess', emittingPayload);
        }),
      )
      .catch(() => {
        this.main.emit('alertLoadFail', {
          hasAlert: false,
          categoryId: this.tree.lastHighlighted,
        });
      });
  }

  public get closestId(): number | undefined {
    if (!this.canBeVisibleAlert) {
      return undefined;
    }

    const { highlightPath } = this.tree;
    let i = highlightPath.length - 1;
    while (i >= 0) {
      if (this.byCategoryId.has(highlightPath[i])) {
        return highlightPath[i];
      }
      i--;
    }
  }

  public get canBeVisibleAlert(): boolean {
    return this.tree.lastHighlighted != null && Boolean(this.byCategoryId.size);
  }

  public get visibleAlert() {
    if (this.closestId != null) {
      return this.byCategoryId.get(this.closestId);
    }
    return undefined;
  }

  public get hasRelatedAlerts() {
    return this.byCategoryId.size > 1 && this.tree.highlightPath.length > 1;
  }

  public get relatedAlerts() {
    if (!this.hasRelatedAlerts) {
      return [];
    }

    return this.tree.highlightPath
      .filter((categoryId) => this.byCategoryId.has(categoryId))
      .map((categoryId) => this.byCategoryId.get(categoryId) as AlertDTO);
  }

  public setExternalHandler(externalHandler: AlertLoadHandler) {
    this.externalHandler = externalHandler;
  }

  public reset() {
    this.externalHandler = () => Promise.resolve([]);
    this.byCategoryId.clear();
    this.expanding.on();
  }
}
