import { makeObservable, observable, action, autorun } from 'mobx';
import debounce from 'lodash/debounce';
import { Store, Search as ISearch, Resetable, SearchExternalHandler } from '../../../types';
import { filterRootIds } from '../../../utils';
import { BoolState } from '../BoolState';

export class Search implements ISearch, Resetable {
  private externalHandler: SearchExternalHandler = () =>
    Promise.resolve({
      resultIds: [],
      highlightRangesById: {},
    });

  public constructor(store?: Store) {
    if (store) {
      this.store = store;
    }

    makeObservable(this, {
      text: observable,
      setText: action.bound,
      resultIds: observable,
      setResultIds: action.bound,
      handler: action.bound,
    });
  }

  public store: Store;

  public setStores(stores: { main: Store }) {
    const { main } = stores;
    this.store = main;
  }

  public setExternalHandler(externalHandler: SearchExternalHandler) {
    this.externalHandler = externalHandler;
  }

  public runReactions() {
    return [
      autorun(() => {
        if (this.store.tabs.previous === 'search' && this.text.length > 2) {
          this.setText('');
        }
      }),
      autorun(() => {
        if (this.store.tabs.previous === 'search') {
          this.loading.on();
          this.setResultIds([]);
          this.store.textHighlighting.setById({});
        }
      }),
    ];
  }

  public text: string = '';

  public setText(text: string) {
    this.text = text;
  }

  public resultIds: number[] = [];

  public setResultIds(ids: number[]) {
    this.resultIds = ids;
  }

  public loading = new BoolState(true);

  public error = new BoolState(false);

  private debouncedHandler = debounce((text: string) => {
    this.externalHandler(text)
      .then(({ resultIds = [], highlightRangesById = {} }) => {
        this.error.off();
        this.setResultIds(filterRootIds(this.store.tree.getById, resultIds || []));
        this.store.textHighlighting.setById(highlightRangesById);
        this.store.emit('search', {
          text,
          resultsCount: this.resultIds.length,
        });
      })
      .catch(() => {
        this.error.on();
      })
      .finally(() => {
        this.loading.off();
      });
  }, 300);

  public handler(text: string) {
    this.setText(text);

    if (this.text.length > 2) {
      this.debouncedHandler(text);
      this.store.tabs.go('search');
    } else {
      this.debouncedHandler.cancel();
      if (this.store.tabs.current === 'search') {
        this.store.tabs.back();
      }
    }
  }

  public reset() {
    this.text = '';
    this.resultIds = [];
    this.loading.on();
    this.error.off();
  }
}
