import { ComponentType } from 'react';
import { makeObservable, observable, action, autorun, computed } from 'mobx';
import isEqual from 'lodash/isEqual';
import { Subject } from 'rxjs';
import {
  Store as IStore,
  ExternalData,
  TargetMeta,
  Tabs as ITabs,
  Search as ISearch,
  Tree as ITree,
  Tip as ITip,
  EmittingPayload,
  SaveHandler,
  TextHighlighting as ITextHighlighting,
  AlertStore as IAlertStore,
  LoadHandler,
  StoreEmitter,
} from '../../../types';
import { Tabs } from '../Tabs';
import { Search } from '../Search';
import { Tree } from '../Tree';
import { Tip } from '../Tip';
import { TextHighlighting } from '../TextHighlighting';
import { AlertStore } from '../AlertStore';
import { ConstructorConfig } from './Store.types';
import { Configurator } from './Configurator';
import { AbstractConfigurator } from '../AbstractConfigurator';
import { BoolState } from '../BoolState';

export class Store implements IStore {
  public constructor(config: ConstructorConfig = {}) {
    const {
      emitter = new Subject(),
      tabs = new Tabs(),
      search = new Search(),
      tree = new Tree(),
      tip = new Tip(),
      textHighlighting = new TextHighlighting(),
      alertStore = new AlertStore(),
      configurator = new Configurator(),
    } = config;

    this.setEmitter(emitter);
    this.tabs = tabs;
    this.tree = tree;
    this.setSearch(search);
    this.setTip(tip);
    this.textHighlighting = textHighlighting;
    this.setAlertStore(alertStore);
    this.setConfigurator(configurator);

    this.emit = this.emit.bind(this);

    makeObservable(this, {
      setup: action.bound,
      targetMeta: observable,
      previewComponent: observable,
      open: action.bound,
      close: action.bound,
      hasChanges: computed,
      save: action.bound,
    });
  }

  private reactionDisposers: (() => void)[] = [];

  public stopReactions() {
    this.reactionDisposers.forEach((disposer) => disposer());
  }

  public runReactions() {
    return [
      autorun(() => {
        if (this.tabs.current === 'selected' && !this.tree.finiteSelected.length) {
          this.tabs.go('tree');
        }
      }),
      ...this.alertStore.runReactions(),
      ...this.tip.runReactions(),
      ...this.search.runReactions(),
    ];
  }

  public emitter: StoreEmitter;

  public setEmitter(emitter: StoreEmitter) {
    this.emitter = emitter;
  }

  public emit<ExtraData extends EmittingPayload>(event: string, extraData?: ExtraData) {
    this.emitter.next({
      event,
      payload: {
        ...this.targetMeta,
        ...extraData,
      },
    });
  }

  public setSearch(search: ISearch) {
    this.search = search;
    this.search.setStores({
      main: this,
    });
  }

  public setTip(tip: ITip) {
    this.tip = tip;
    this.tip.setStores({
      main: this,
    });
  }

  public setAlertStore(alertStore: IAlertStore) {
    this.alertStore = alertStore;
    this.alertStore.setStores({
      tree: this.tree,
      main: this,
    });
  }

  public loadHandler?: LoadHandler;

  public setLoadHandler(loadHandler: LoadHandler) {
    this.loadHandler = loadHandler;
  }

  public saveHandler: SaveHandler = () => {};

  public setSaveHandler(saveHandler: SaveHandler) {
    this.saveHandler = saveHandler;
  }

  public previewComponent: ComponentType;

  private configurator: AbstractConfigurator<[ExternalData?]>;

  public setConfigurator(configurator: AbstractConfigurator<[ExternalData?]>) {
    this.configurator = configurator;
    this.configurator.setConfigurable(this);
  }

  public setup(externalData?: ExternalData) {
    this.configurator.setup(externalData);
  }

  public targetMeta: TargetMeta;

  public tabs: ITabs;

  public search: ISearch;

  public tree: ITree;

  public tip: ITip;

  public textHighlighting: ITextHighlighting;

  public alertStore: IAlertStore;

  public openess = new BoolState(false);

  public fullness = new BoolState(true);

  public close() {
    this.reset();
  }

  public open(externalData?: ExternalData) {
    this.reset();
    this.setup(externalData);
    this.openess.on();

    let promise: Promise<unknown> = Promise.resolve();

    if (this.loadHandler) {
      this.tree.loading.on();
      promise = this.loadHandler()
        .then((data) => {
          this.tree.setup(data);
        })
        .finally(() => {
          this.tree.loading.off();
        });
    }

    promise.finally(() => {
      if (this.tree.finiteSelected.length) {
        this.tabs.go('selected');
      }
    });

    this.reactionDisposers = this.runReactions();
  }

  public get hasChanges(): boolean {
    return !isEqual(this.tree.valueAsTree.initial, this.tree.valueAsTree.current);
  }

  public save() {
    this.saveHandler(
      this.tree.finiteSelected.map((id) => ({
        id,
      })),
    );
    this.close();
  }

  public reset() {
    this.search.reset();
    this.tree.reset();
    this.tabs.reset();
    this.tip.reset();
    this.textHighlighting.reset();
    this.alertStore.reset();
    this.stopReactions();
    this.reactionDisposers = [];
    this.fullness.on();
    this.openess.off();
  }
}
