import { makeObservable, observable, action, autorun } from 'mobx';
import {
  Rating as IRating,
  Store,
  CommentFormExternalHandler,
  CommentForm as ICommentForm,
} from '../../../../types';
import { ById } from './Rating.types';
import { SESSION_STORAGE_KEY } from './Rating.constants';
import { CommentForm } from './CommentForm';

export class Rating implements IRating {
  public byId: ById = {};

  public store: Store;

  public constructor(commentForm: ICommentForm = new CommentForm()) {
    makeObservable(this, {
      byId: observable,
      like: action.bound,
      dislike: action.bound,
      loadFromSessionStorage: action.bound,
    });

    this.commentForm = commentForm;
  }

  public runReactions() {
    return [
      autorun(() => {
        if (this.store.openess.state) {
          this.loadFromSessionStorage();
        }
      }),
      autorun(() => {
        this.saveInSessionStorage(this.byId);
      }),
    ];
  }

  public commentForm: ICommentForm;

  public setExternalHandler(externalHandler: CommentFormExternalHandler) {
    this.commentForm.setExternalHandler(externalHandler);
  }

  public like(id: number) {
    if (this.byId[id]) {
      this.byId[id] = undefined;
    } else {
      this.byId[id] = true;
    }
  }

  public dislike(id: number) {
    if (this.byId[id] || this.byId[id] === undefined) {
      this.byId[id] = false;
    } else {
      this.byId[id] = undefined;
    }
  }

  public setStores(stores: { main: Store }) {
    const { main } = stores;
    this.store = main;
    this.commentForm.setStores({
      main,
      rating: this,
    });
  }

  public reset() {
    this.byId = {};
    this.commentForm.reset();
  }

  private saveInSessionStorage(byId: ById) {
    sessionStorage.setItem(SESSION_STORAGE_KEY, JSON.stringify(byId));
  }

  public loadFromSessionStorage() {
    const saved: ById = JSON.parse(sessionStorage.getItem(SESSION_STORAGE_KEY) || '{}');

    if (saved) {
      this.byId = saved;
    }
  }
}
