import { makeObservable, observable, action, reaction } from 'mobx';
import { config } from 'services/Config';
import { Tip as ITip, Store, TipExternalHandler, Rating as IRating } from '../../../types';
import { Rating } from './Rating';
import { BoolState } from '../BoolState';
export class Tip implements ITip {
  public store: Store;

  private externalHandler: TipExternalHandler = () => Promise.resolve('');

  public loading = new BoolState(true);

  public html?: string;

  public rating: IRating;

  public constructor(rating: IRating = new Rating()) {
    makeObservable(this, {
      html: observable,
      setHtml: action.bound,
      load: action.bound,
      reset: action.bound,
    });

    this.rating = rating;
  }

  public setExternalHandler(externalHandler: TipExternalHandler) {
    this.externalHandler = externalHandler;
  }

  public setHtml(html?: string) {
    this.html = html;
  }

  public load(id: number) {
    const { emit, tabs } = this.store;
    this.loading.on();
    this.externalHandler(id)
      .then((html) => {
        if (html.length) {
          this.setHtml(html);
        } else {
          this.setHtml(undefined);
        }

        emit('tipLoadSuccess', {
          block: tabs.current,
          categoryId: id,
          hasTip: Boolean(html.length),
        });
      })
      .catch(() => {
        this.setHtml(undefined);

        emit('tipLoadFail', {
          block: tabs.current,
          categoryId: id,
        });
      })
      .finally(() => {
        this.loading.off();
      });
  }

  public runReactions() {
    return [
      reaction(
        () => this.store.tree.lastHighlighted,
        (lastHighlighted) => {
          if (lastHighlighted == null || config.value.features.categoryTipsExperiment) {
            this.setHtml(undefined);
            return;
          }

          this.load(lastHighlighted);
        },
      ),
      ...this.rating.runReactions(),
    ];
  }

  public setStores(stores: { main: Store }) {
    const { main } = stores;
    this.store = main;
    this.rating.setStores({
      main,
    });
  }

  public reset() {
    this.setHtml(undefined);
    this.loading.on();
    this.rating.reset();
  }
}
